// MixNET - 网络仿真软件核心代码

// 全局变量
let canvas;
let devices = [];
let connections = [];
let selectedDevice = null;
let selectedConnection = null;
let isConnecting = false;
let startDevice = null;
let connectionCounter = 0;
let isSimulating = false;
let simulationStartTime = null;
let packetCount = 0;
let simulationInterval = null;

// 设备类型定义
const DEVICE_TYPES = {
    computer: {
        name: "台式机",
        icon: "🖥️",
        width: 60,
        height: 60,
        defaultProps: {
            ip: "192.168.1.100",
            subnetMask: "255.255.255.0",
            gateway: "192.168.1.1",
            dns: "8.8.8.8"
        }
    },
    laptop: {
        name: "笔记本",
        icon: "💻",
        width: 60,
        height: 60,
        defaultProps: {
            ip: "192.168.1.101",
            subnetMask: "255.255.255.0",
            gateway: "192.168.1.1",
            dns: "8.8.8.8",
            ipMode: "dhcp", // dhcp 或 static
            ssid: "MixNet_WiFi",
            password: "",
            wirelessEnabled: true
        }
    },
    router: {
        name: "路由器",
        icon: "<svg width='32' height='32' viewBox='0 0 32 32' xmlns='http://www.w3.org/2000/svg'><rect width='28' height='16' x='2' y='8' rx='2' fill='#5c6bc0'/><rect width='26' height='12' x='3' y='10' rx='1' fill='#3f51b5'/><rect width='2' height='4' x='5' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='8' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='11' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='14' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='17' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='20' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='23' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='26' y='14' rx='1' fill='#7986cb'/><circle cx='7' cy='12' r='1' fill='#4caf50'/><circle cx='13' cy='12' r='1' fill='#ffeb3b'/><circle cx='19' cy='12' r='1' fill='#ff9800'/><circle cx='25' cy='12' r='1' fill='#f44336'/><rect width='22' height='2' x='5' y='22' rx='1' fill='#9fa8da'/></svg>",
        width: 80,
        height: 80,
        defaultProps: {
            interfaces: [
                { name: "eth0", ip: "192.168.1.1", subnetMask: "255.255.255.0" },
                { name: "eth1", ip: "10.0.0.1", subnetMask: "255.255.255.0" },
                { name: "eth2", ip: "172.16.0.1", subnetMask: "255.255.255.0" }
            ],
            routingTable: [
                // 默认路由表项格式: { network: "目标网络", subnetMask: "子网掩码", nextHop: "下一跳IP", interface: "出接口" }
            ]
        }
    },
    switch: {
        name: "交换机",
        icon: "🔄",
        width: 70,
        height: 50,
        defaultProps: {
            ports: 8,
            vlan: {
                enabled: false,
                vlans: []
            }
        }
    },
    "dns-server": {
        name: "DNS服务器",
        icon: "📡",
        width: 60,
        height: 60,
        defaultProps: {
            ip: "192.168.1.50",
            subnetMask: "255.255.255.0",
            domains: [
                { name: "example.com", ip: "192.168.1.60", type: "A" },
                { name: "test.org", ip: "192.168.1.61", type: "A" }
            ]
        }
    },
    "web-server": {
        name: "Web服务器",
        icon: "🌐",
        width: 60,
        height: 60,
        defaultProps: {
            ip: "192.168.1.200",
            subnetMask: "255.255.255.0",
            website: {
                title: "欢迎页面",
                content: "这是一个简单的Web服务器页面"
            }
        }
    },
    "email-server": {
        name: "Email服务器",
        icon: "📧",
        width: 60,
        height: 60,
        defaultProps: {
            ip: "192.168.1.210",
            subnetMask: "255.255.255.0",
            domain: "example.com",
            smtpPort: 25,
            pop3Port: 110,
            imapPort: 143,
            users: [
                { username: "user1", password: "password1", emails: [] },
                { username: "user2", password: "password2", emails: [] }
            ],
            emails: []
        }
    },
    "wireless-router": {
        name: "WiFi路由器",
        icon: "📶",
        width: 80,
        height: 80,
        defaultProps: {
            interfaces: [
                { name: "eth0", ip: "10.0.0.1", subnetMask: "255.255.255.0", role: "wan" }, // 外网接口
                { name: "wlan0", ip: "192.168.1.1", subnetMask: "255.255.255.0", role: "lan" }  // 内网接口（无线）
            ],
            ssid: "MixNet_WiFi",
            security: "WPA2-PSK",
            password: "password123",
            natEnabled: true, // 启用NAT
            routingTable: [] // 默认没有路由表，可以访问直连网络
        }
    }
};

// 初始化应用
function init() {
    // 初始化SVG画布
    const svgElement = document.getElementById('network-canvas');
    canvas = svgElement;
    
    // 将devices暴露到window对象，以便在动态生成的脚本中访问
    window.devices = devices;
    
    // 添加事件监听器
    addEventListeners();
    
    // 初始化模态对话框
    initModal();
    
    // 记录日志
    log("应用初始化完成");
    
    // 初始更新WiFi连接虚线
    updateWirelessConnections();
    
    // 初始化实验工具状态
    updateExperimentToolsStatus();
    
    // 将画布初始位置居中
    centerCanvas();
}

// 添加事件监听器
// 触摸拖拽相关变量
let isTouchDragging = false;
let touchDeviceType = null;
let touchStartX = 0;
let touchStartY = 0;
let dragPreview = null;

// 全局拖拽状态管理
let isDragging = false;
let draggedElement = null;
let elementX = 0;
let elementY = 0;
let startX = 0;
let startY = 0;

// 处理设备拖拽开始事件
function handleDeviceDragStart(e) {
    const deviceType = this.dataset.device;
    e.dataTransfer.setData('text/plain', deviceType);
    // 设置拖拽效果
    e.dataTransfer.effectAllowed = 'copy';
}

// 处理画布拖拽经过事件
function handleCanvasDragOver(e) {
    e.preventDefault(); // 允许放置
    e.dataTransfer.dropEffect = 'copy'; // 设置放置效果
}

// 处理画布拖拽放置事件
function handleCanvasDrop(e) {
    e.preventDefault(); // 阻止默认行为
    
    const deviceType = e.dataTransfer.getData('text/plain');
    if (!deviceType) return;
    
    // 计算放置位置相对于画布的坐标
    const canvasWrapper = document.getElementById('canvas-wrapper');
    const rect = canvasWrapper.getBoundingClientRect();
    const x = e.clientX - rect.left + canvasWrapper.scrollLeft;
    const y = e.clientY - rect.top + canvasWrapper.scrollTop;
    
    // 添加新设备
    addDevice(deviceType, x, y);
}

// 将画布初始位置居中
function centerCanvas() {
    const canvasContainer = document.querySelector('.canvas-container');
    const canvasWrapper = document.getElementById('canvas-wrapper');
    
    if (!canvasContainer || !canvasWrapper) return;
    
    // 使用requestAnimationFrame确保在下一帧执行，此时容器尺寸已确定
    requestAnimationFrame(() => {
        const canvasWidth = 2000; // 画布总宽度
        const canvasHeight = 2000; // 画布总高度
        const containerWidth = canvasContainer.clientWidth;
        const containerHeight = canvasContainer.clientHeight;
        
        // 如果容器尺寸还未确定，延迟执行
        if (containerWidth === 0 || containerHeight === 0) {
            setTimeout(() => centerCanvas(), 50);
            return;
        }
        
        // 计算居中位置
        const centerX = Math.max(0, (canvasWidth - containerWidth) / 2);
        const centerY = Math.max(0, (canvasHeight - containerHeight) / 2);
        
        canvasContainer.scrollLeft = centerX;
        canvasContainer.scrollTop = centerY;
    });
}

// 画布拖动滚动功能
let isCanvasPanning = false;
let panStartX = 0;
let panStartY = 0;
let panScrollLeft = 0;
let panScrollTop = 0;

function initCanvasPan() {
    const canvasContainer = document.querySelector('.canvas-container');
    const canvasWrapper = document.getElementById('canvas-wrapper');
    
    if (!canvasContainer || !canvasWrapper) return;
    
    // 鼠标按下事件
    canvasContainer.addEventListener('mousedown', function(e) {
        // 如果点击的是设备或连接线，不触发画布拖动
        if (e.target.closest('.device') || e.target.closest('.connection') || 
            e.target.closest('.connection-dot') || e.target.closest('g[class*="device"]') ||
            e.target.closest('g') || e.target.tagName === 'g') {
            return;
        }
        
        // 如果正在连接设备，不触发画布拖动
        if (isConnecting) {
            return;
        }
        
        // 如果正在拖动设备，不触发画布拖动
        if (isDragging) {
            return;
        }
        
        isCanvasPanning = true;
        canvasContainer.classList.add('dragging');
        
        panStartX = e.clientX;
        panStartY = e.clientY;
        panScrollLeft = canvasContainer.scrollLeft;
        panScrollTop = canvasContainer.scrollTop;
        
        e.preventDefault();
    });
    
    // 鼠标移动事件
    document.addEventListener('mousemove', function(e) {
        if (!isCanvasPanning) return;
        
        // 如果开始拖动设备，停止画布拖动
        if (isDragging) {
            isCanvasPanning = false;
            canvasContainer.classList.remove('dragging');
            return;
        }
        
        const deltaX = panStartX - e.clientX;
        const deltaY = panStartY - e.clientY;
        
        canvasContainer.scrollLeft = panScrollLeft + deltaX;
        canvasContainer.scrollTop = panScrollTop + deltaY;
        
        e.preventDefault();
    });
    
    // 鼠标释放事件
    document.addEventListener('mouseup', function(e) {
        if (isCanvasPanning) {
            isCanvasPanning = false;
            canvasContainer.classList.remove('dragging');
        }
    });
    
    // 鼠标离开窗口时也停止拖动
    document.addEventListener('mouseleave', function(e) {
        if (isCanvasPanning) {
            isCanvasPanning = false;
            canvasContainer.classList.remove('dragging');
        }
    });
}

function addEventListeners() {
    // 设备事件
    document.querySelectorAll('.device-item').forEach(item => {
        // 保留 draggable 属性，以便将来可能的功能扩展
        item.setAttribute('draggable', true);
        
        // 添加拖拽开始事件
        item.addEventListener('dragstart', handleDeviceDragStart);
        
        // 添加点击事件，用于在画布中央添加新设备
        item.addEventListener('click', function() {
            const deviceType = this.dataset.device;
            // 在画布中央添加新设备
            const canvasWidth = 2000;
            const canvasHeight = 2000;
            const centerX = canvasWidth / 2;
            const centerY = canvasHeight / 2;
            addDevice(deviceType, centerX, centerY);
        });
    });
    
    // 画布拖拽放置事件
    const canvasWrapper = document.getElementById('canvas-wrapper');
    canvasWrapper.addEventListener('dragover', handleCanvasDragOver);
    canvasWrapper.addEventListener('drop', handleCanvasDrop);
    
    // 画布拖动滚动功能
    initCanvasPan();
    
    // 导航栏事件
    document.getElementById('new-topology').addEventListener('click', newTopology);
    document.getElementById('open-topology').addEventListener('click', openTopology);
    document.getElementById('save-topology').addEventListener('click', saveTopology);
    document.getElementById('toggle-simulation').addEventListener('click', toggleSimulation);
    document.getElementById('help').addEventListener('click', showHelp);
    
    // 连接工具事件
    document.getElementById('connect-devices').addEventListener('click', toggleConnectMode);
    document.getElementById('delete-connection').addEventListener('click', deleteConnection);
    
    // 实验工具事件
    document.getElementById('ping-tool').addEventListener('click', showPingTool);
    document.getElementById('traceroute-tool').addEventListener('click', showTracerouteTool);
    document.getElementById('nslookup-tool').addEventListener('click', showNSLookupTool);
    document.getElementById('web-browser').addEventListener('click', showWebBrowser);
    document.getElementById('email-tool').addEventListener('click', showEmailTool);
    document.getElementById('network-command-tool').addEventListener('click', showNetworkCommandTool);
    
    // 预设实验事件
    document.querySelectorAll('.experiment-item').forEach(item => {
        item.addEventListener('click', () => {
            loadPresetExperiment(item.dataset.experiment);
        });
    });
    
    // 原理展示事件
    document.querySelectorAll('.principle-item').forEach(item => {
        item.addEventListener('click', () => {
            showPrinciple(item.id);
        });
    });
    
    // 保存所有定时器的ID，用于清除
    let principleTimers = [];
    
    // 控制动画是否应该继续播放的标志
    let isAnimating = false;
    
    // 三次握手原理展示函数
    function showThreeWayHandshake() {
        const modal = document.getElementById('modal');
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <div class="modal-header">
                <h3>TCP三次握手原理</h3>
            </div>
            <div class="property-form">
                <!-- 动画展示区域 -->
                <div class="form-group">
                    <label>TCP三次握手过程</label>
                    <div id="three-way-handshake-animation" style="
                        background-color: #f8f9fa;
                        padding: 15px;
                        border-radius: 8px;
                        margin-top: 10px;
                        min-height: 400px;
                        max-height: 500px;
                        overflow-y: auto;
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        width: 100%;
                    ">
                        <!-- 发送端和接收端 -->
                        <div style="
                            display: flex;
                            gap: 60px;
                            align-items: flex-start;
                            position: relative;
                            width: 100%;
                            justify-content: center;
                        ">
                            <!-- 客户端 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">客户端</h5>
                                <div id="client-device" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #3498db;
                                    border-radius: 8px;
                                    padding: 15px;
                                    width: 150px;
                                    min-height: 120px;
                                    display: flex;
                                    flex-direction: column;
                                    align-items: center;
                                    justify-content: center;
                                ">
                                    <div style="font-size: 40px; margin-bottom: 5px;">💻</div>
                                    <div style="font-size: 12px;">Client</div>
                                </div>
                            </div>
                            
                            <!-- 网络连接 -->
                            <div style="
                                width: 1px;
                                height: 200px;
                                background-color: #bdc3c7;
                                position: relative;
                            ">
                                <!-- SYN包 -->
                                <div id="syn-packet" style="
                                    position: absolute;
                                    top: 10px;
                                    left: -25px;
                                    width: 50px;
                                    height: 20px;
                                    background-color: rgba(46, 204, 113, 0.9);
                                    border: 2px solid #27ae60;
                                    border-radius: 4px;
                                    display: flex;
                                    align-items: center;
                                    justify-content: center;
                                    font-size: 11px;
                                    font-weight: bold;
                                    color: white;
                                    z-index: 10;
                                    opacity: 0;
                                ">SYN</div>
                                
                                <!-- SYN-ACK包 -->
                                <div id="syn-ack-packet" style="
                                    position: absolute;
                                    top: 60px;
                                    left: -30px;
                                    width: 60px;
                                    height: 20px;
                                    background-color: rgba(255, 193, 7, 0.9);
                                    border: 2px solid #ff9800;
                                    border-radius: 4px;
                                    display: flex;
                                    align-items: center;
                                    justify-content: center;
                                    font-size: 11px;
                                    font-weight: bold;
                                    color: white;
                                    z-index: 10;
                                    opacity: 0;
                                    white-space: nowrap;
                                ">SYN-ACK</div>
                                
                                <!-- ACK包 -->
                                <div id="ack-packet" style="
                                    position: absolute;
                                    top: 110px;
                                    left: -25px;
                                    width: 50px;
                                    height: 20px;
                                    background-color: rgba(231, 76, 60, 0.9);
                                    border: 2px solid #e74c3c;
                                    border-radius: 4px;
                                    display: flex;
                                    align-items: center;
                                    justify-content: center;
                                    font-size: 11px;
                                    font-weight: bold;
                                    color: white;
                                    z-index: 10;
                                    opacity: 0;
                                ">ACK</div>
                            </div>
                            
                            <!-- 服务器 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">服务器</h5>
                                <div id="server-device" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #e74c3c;
                                    border-radius: 8px;
                                    padding: 15px;
                                    width: 150px;
                                    min-height: 120px;
                                    display: flex;
                                    flex-direction: column;
                                    align-items: center;
                                    justify-content: center;
                                ">
                                    <div style="font-size: 40px; margin-bottom: 5px;">🖥️</div>
                                    <div style="font-size: 12px;">Server</div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 状态说明 -->
                        <div id="handshake-status" style="
                            margin-top: 20px;
                            padding: 10px;
                            background-color: #f0f3f4;
                            border-radius: 4px;
                            width: 100%;
                            text-align: center;
                            font-size: 14px;
                            font-weight: bold;
                            color: #34495e;
                        ">TCP三次握手开始...</div>
                    </div>
                    
                    <!-- 控制按钮 -->
                    <div class="form-actions" style="margin-top: 15px; display: flex; gap: 10px; justify-content: center;">
                        <button id="start-handshake-btn" class="btn-confirm">开始</button>
                        <button id="close-handshake-btn" class="btn-cancel">关闭</button>
                    </div>
            </div>
        `;
        
        modal.style.display = 'block';
        initModalDrag();
        
        // 添加事件监听器
        const startBtn = document.getElementById('start-handshake-btn');
        const closeBtn = document.getElementById('close-handshake-btn');
        
        if (startBtn) {
            startBtn.addEventListener('click', startThreeWayHandshakeAnimation);
        }
        
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                document.getElementById('modal').style.display = 'none';
            });
        }
    }
    
    // 三次握手动画函数
    function startThreeWayHandshakeAnimation() {
        console.log('开始三次握手动画');
        isAnimating = true;
        
        // 禁用开始按钮
        const startBtn = document.getElementById('start-handshake-btn');
        if (startBtn) {
            startBtn.disabled = true;
            startBtn.style.opacity = '0.5';
            startBtn.style.cursor = 'not-allowed';
        }
        
        const synPacket = document.getElementById('syn-packet');
        const synAckPacket = document.getElementById('syn-ack-packet');
        const ackPacket = document.getElementById('ack-packet');
        const status = document.getElementById('handshake-status');
        const clientDevice = document.getElementById('client-device');
        const serverDevice = document.getElementById('server-device');
        
        // 重置所有状态
        [synPacket, synAckPacket, ackPacket].forEach(packet => {
            packet.style.opacity = '0';
            packet.style.transform = 'translateX(0)';
        });
        
        // 步骤1: 客户端发送SYN包
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '客户端发送SYN包（同步请求）';
            clientDevice.style.borderColor = '#27ae60';
            clientDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            
            synPacket.style.opacity = '1';
            synPacket.style.transform = 'translateX(80px)';
            synPacket.style.transition = 'all 1s ease-in-out';
        }, 1000));
        
        // 步骤2: 服务器回复SYN-ACK包
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '服务器回复SYN-ACK包（同步确认）';
            clientDevice.style.borderColor = '#3498db';
            clientDevice.style.boxShadow = 'none';
            serverDevice.style.borderColor = '#ff9800';
            serverDevice.style.boxShadow = '0 0 10px rgba(255, 193, 7, 0.5)';
            
            synAckPacket.style.opacity = '1';
            synAckPacket.style.transform = 'translateX(-80px)';
            synAckPacket.style.transition = 'all 1s ease-in-out';
        }, 3000));
        
        // 步骤3: 客户端发送ACK包
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '客户端发送ACK包（确认）';
            serverDevice.style.borderColor = '#e74c3c';
            serverDevice.style.boxShadow = 'none';
            clientDevice.style.borderColor = '#3498db';
            clientDevice.style.boxShadow = 'none';
            
            ackPacket.style.opacity = '1';
            ackPacket.style.transform = 'translateX(80px)';
            ackPacket.style.transition = 'all 1s ease-in-out';
        }, 5000));
        
        // 完成状态
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '三次握手完成，TCP连接建立！';
            status.style.backgroundColor = '#27ae60';
            status.style.color = 'white';
            
            // 让所有设备边框高亮
            clientDevice.style.borderColor = '#27ae60';
            serverDevice.style.borderColor = '#27ae60';
            clientDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            serverDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            
            // 动画结束，启用开始按钮
            const startBtn = document.getElementById('start-handshake-btn');
            if (startBtn) {
                startBtn.disabled = false;
                startBtn.style.opacity = '1';
                startBtn.style.cursor = 'pointer';
            }
            isAnimating = false;
        }, 7000));
    }
    
    // 滑动窗口原理展示函数
    function showSlidingWindow() {
        const modal = document.getElementById('modal');
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <div class="modal-header">
                <h3>TCP滑动窗口原理</h3>
            </div>
            <div class="property-form">
                <!-- 动画展示区域 -->
                <div class="form-group">
                    <label>TCP滑动窗口过程</label>
                    <div id="sliding-window-animation" style="
                        background-color: #f8f9fa;
                        padding: 15px;
                        border-radius: 8px;
                        margin-top: 10px;
                        min-height: 400px;
                        max-height: 500px;
                        overflow-y: auto;
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        width: 100%;
                    ">
                        <!-- 发送端和接收端 -->
                        <div style="
                            display: flex;
                            gap: 60px;
                            align-items: flex-start;
                            position: relative;
                            width: 100%;
                            justify-content: center;
                        ">
                            <!-- 发送端 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">发送端</h5>
                                <div id="sender-device" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #3498db;
                                    border-radius: 8px;
                                    padding: 15px;
                                    width: 150px;
                                    min-height: 120px;
                                    display: flex;
                                    flex-direction: column;
                                    align-items: center;
                                    justify-content: center;
                                ">
                                    <div style="font-size: 40px; margin-bottom: 5px;">💻</div>
                                    <div style="font-size: 12px;">Sender</div>
                                </div>
                                
                                <!-- 发送窗口 -->
                                <div id="sender-window" style="
                                    margin-top: 15px;
                                    width: 200px;
                                    height: 40px;
                                    border: 2px solid #3498db;
                                    border-radius: 4px;
                                    position: relative;
                                    background-color: white;
                                ">
                                    <!-- 发送窗口的数据包 -->
                                    ${Array.from({length: 10}, (_, i) => `
                                        <div id="sender-packet-${i}" style="
                                            position: absolute;
                                            top: 2px;
                                            left: ${i * 20 + 2}px;
                                            width: 16px;
                                            height: 32px;
                                            background-color: ${i < 3 ? 'rgba(46, 204, 113, 0.9)' : '#e0e0e0'};
                                            border: 2px solid ${i < 3 ? '#27ae60' : '#bdc3c7'};
                                            border-radius: 3px;
                                            display: flex;
                                            align-items: center;
                                            justify-content: center;
                                            font-size: 9px;
                                            font-weight: bold;
                                            color: ${i < 3 ? 'white' : '#7f8c8d'};
                                        ">${i}</div>
                                    `).join('')}
                                    
                                    <!-- 发送窗口边框 -->
                                    <div id="sender-window-border" style="
                                        position: absolute;
                                        top: 0;
                                        left: ${3 * 20}px;
                                        width: ${7 * 20}px;
                                        height: 40px;
                                        border: 2px dashed #3498db;
                                        border-radius: 4px;
                                        z-index: 10;
                                    "></div>
                                </div>
                                <div style="margin-top: 5px; font-size: 10px; color: #7f8c8d;">窗口大小: 3</div>
                            </div>
                            
                            <!-- 网络连接 -->
                            <div style="
                                width: 1px;
                                height: 250px;
                                background-color: #bdc3c7;
                                position: relative;
                            ">
                                <!-- 发送的数据包 -->
                                ${Array.from({length: 3}, (_, i) => `
                                    <div id="sent-packet-${i}" style="
                                        position: absolute;
                                        top: ${10 + i * 30}px;
                                        left: -25px;
                                        width: 50px;
                                        height: 20px;
                                        background-color: rgba(46, 204, 113, 0.9);
                                        border: 2px solid #27ae60;
                                        border-radius: 4px;
                                        display: flex;
                                        align-items: center;
                                        justify-content: center;
                                        font-size: 11px;
                                        font-weight: bold;
                                        color: white;
                                        z-index: 10;
                                        opacity: 0;
                                    ">Pkt ${i}</div>
                                `).join('')}
                                
                                <!-- 确认包 -->
                                <div id="ack-packet" style="
                                    position: absolute;
                                    top: 150px;
                                    left: -25px;
                                    width: 50px;
                                    height: 20px;
                                    background-color: rgba(255, 193, 7, 0.9);
                                    border: 2px solid #ff9800;
                                    border-radius: 4px;
                                    display: flex;
                                    align-items: center;
                                    justify-content: center;
                                    font-size: 11px;
                                    font-weight: bold;
                                    color: white;
                                    z-index: 10;
                                    opacity: 0;
                                ">ACK 3</div>
                            </div>
                            
                            <!-- 接收端 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">接收端</h5>
                                <div id="receiver-device" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #e74c3c;
                                    border-radius: 8px;
                                    padding: 15px;
                                    width: 150px;
                                    min-height: 120px;
                                    display: flex;
                                    flex-direction: column;
                                    align-items: center;
                                    justify-content: center;
                                ">
                                    <div style="font-size: 40px; margin-bottom: 5px;">🖥️</div>
                                    <div style="font-size: 12px;">Receiver</div>
                                </div>
                                
                                <!-- 接收窗口 -->
                                <div id="receiver-window" style="
                                    margin-top: 15px;
                                    width: 200px;
                                    height: 40px;
                                    border: 2px solid #e74c3c;
                                    border-radius: 4px;
                                    position: relative;
                                    background-color: white;
                                ">
                                    <!-- 接收窗口的数据包 -->
                                    ${Array.from({length: 10}, (_, i) => `
                                        <div id="receiver-packet-${i}" style="
                                            position: absolute;
                                            top: 2px;
                                            left: ${i * 20 + 2}px;
                                            width: 16px;
                                            height: 32px;
                                            background-color: #e0e0e0;
                                            border: 2px solid #bdc3c7;
                                            border-radius: 3px;
                                            display: flex;
                                            align-items: center;
                                            justify-content: center;
                                            font-size: 9px;
                                            font-weight: bold;
                                            color: #7f8c8d;
                                        ">${i}</div>
                                    `).join('')}
                                </div>
                                <div style="margin-top: 5px; font-size: 10px; color: #7f8c8d;">窗口大小: 5</div>
                            </div>
                        </div>
                        
                        <!-- 状态说明 -->
                        <div id="window-status" style="
                            margin-top: 20px;
                            padding: 10px;
                            background-color: #f0f3f4;
                            border-radius: 4px;
                            width: 100%;
                            text-align: center;
                            font-size: 14px;
                            font-weight: bold;
                            color: #34495e;
                        ">TCP滑动窗口开始...</div>
                    </div>
                    
                    <!-- 控制按钮 -->
                    <div class="form-actions" style="margin-top: 15px; display: flex; gap: 10px; justify-content: center;">
                        <button id="start-window-btn" class="btn-confirm">开始</button>
                        <button id="close-window-btn" class="btn-cancel">关闭</button>
                    </div>
                </div>
            </div>
        `;
        
        modal.style.display = 'block';
        initModalDrag();
        
        // 添加事件监听器
        const startBtn = document.getElementById('start-window-btn');
        const closeBtn = document.getElementById('close-window-btn');
        
        if (startBtn) {
            startBtn.addEventListener('click', startSlidingWindowAnimation);
        }
        
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                document.getElementById('modal').style.display = 'none';
            });
        }
    }
    
    // 滑动窗口动画函数
    function startSlidingWindowAnimation() {
        console.log('开始滑动窗口动画');
        isAnimating = true;
        
        // 禁用开始按钮
        const startBtn = document.getElementById('start-window-btn');
        if (startBtn) {
            startBtn.disabled = true;
            startBtn.style.opacity = '0.5';
            startBtn.style.cursor = 'not-allowed';
        }
        
        const status = document.getElementById('window-status');
        const senderDevice = document.getElementById('sender-device');
        const receiverDevice = document.getElementById('receiver-device');
        const senderWindowBorder = document.getElementById('sender-window-border');
        
        // 重置所有数据包状态
        for (let i = 0; i < 10; i++) {
            const senderPacket = document.getElementById(`sender-packet-${i}`);
            const receiverPacket = document.getElementById(`receiver-packet-${i}`);
            
            if (senderPacket) {
                senderPacket.style.backgroundColor = i < 3 ? 'rgba(46, 204, 113, 0.9)' : '#e0e0e0';
                senderPacket.style.borderColor = i < 3 ? '#27ae60' : '#bdc3c7';
                senderPacket.style.color = i < 3 ? 'white' : '#7f8c8d';
            }
            
            if (receiverPacket) {
                receiverPacket.style.backgroundColor = '#e0e0e0';
                receiverPacket.style.borderColor = '#bdc3c7';
                receiverPacket.style.color = '#7f8c8d';
            }
        }
        
        // 重置发送窗口边框
        senderWindowBorder.style.left = `${3 * 20}px`;
        senderWindowBorder.style.width = `${7 * 20}px`;
        senderWindowBorder.style.transition = 'none';
        
        // 步骤1: 发送方发送前3个数据包(0-2)
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '发送方发送窗口中的3个数据包(0-2)';
            senderDevice.style.borderColor = '#27ae60';
            senderDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            
            // 发送第一个数据包
            const sentPacket0 = document.getElementById('sent-packet-0');
            sentPacket0.style.opacity = '1';
            sentPacket0.style.transform = 'translateX(80px)';
            sentPacket0.style.transition = 'all 1s ease-in-out';
            
            // 发送第二个数据包
            principleTimers.push(setTimeout(() => {
                const sentPacket1 = document.getElementById('sent-packet-1');
                sentPacket1.style.opacity = '1';
                sentPacket1.style.transform = 'translateX(80px)';
                sentPacket1.style.transition = 'all 1s ease-in-out';
            }, 500));
            
            // 发送第三个数据包
            principleTimers.push(setTimeout(() => {
                const sentPacket2 = document.getElementById('sent-packet-2');
                sentPacket2.style.opacity = '1';
                sentPacket2.style.transform = 'translateX(80px)';
                sentPacket2.style.transition = 'all 1s ease-in-out';
            }, 1000));
        }, 1000));
        
        // 步骤2: 接收方接收并确认数据包0-2
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '接收方接收数据包0-2并发送确认ACK 3';
            senderDevice.style.borderColor = '#3498db';
            senderDevice.style.boxShadow = 'none';
            receiverDevice.style.borderColor = '#ff9800';
            receiverDevice.style.boxShadow = '0 0 10px rgba(255, 193, 7, 0.5)';
            
            // 更新接收窗口的数据包状态
            for (let i = 0; i < 3; i++) {
                const receiverPacket = document.getElementById(`receiver-packet-${i}`);
                receiverPacket.style.backgroundColor = 'rgba(46, 204, 113, 0.9)';
                receiverPacket.style.borderColor = '#27ae60';
                receiverPacket.style.color = 'white';
            }
            
            // 发送确认包
            const ackPacket = document.getElementById('ack-packet');
            ackPacket.innerText = 'ACK 3';
            ackPacket.style.opacity = '1';
            ackPacket.style.transform = 'translateX(-80px)';
            ackPacket.style.transition = 'all 1s ease-in-out';
        }, 4000));
        
        // 步骤3: 发送方滑动窗口，发送数据包3-5
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '发送方滑动窗口，发送新的数据包3-5';
            receiverDevice.style.borderColor = '#e74c3c';
            receiverDevice.style.boxShadow = 'none';
            senderDevice.style.borderColor = '#27ae60';
            senderDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            
            // 滑动发送窗口
            senderWindowBorder.style.left = `${6 * 20}px`;
            senderWindowBorder.style.width = `${4 * 20}px`;
            senderWindowBorder.style.transition = 'all 0.5s ease-in-out';
            
            // 更新发送窗口的数据包状态
            for (let i = 0; i < 3; i++) {
                const senderPacket = document.getElementById(`sender-packet-${i}`);
                senderPacket.style.backgroundColor = 'rgba(52, 152, 219, 0.9)';
                senderPacket.style.borderColor = '#2980b9';
            }
            
            for (let i = 3; i < 6; i++) {
                const senderPacket = document.getElementById(`sender-packet-${i}`);
                senderPacket.style.backgroundColor = 'rgba(46, 204, 113, 0.9)';
                senderPacket.style.borderColor = '#27ae60';
                senderPacket.style.color = 'white';
            }
            
            // 发送新的数据包
            setTimeout(() => {
                const sentPacket0 = document.getElementById('sent-packet-0');
                sentPacket0.innerText = 'Pkt 3';
                sentPacket0.style.opacity = '0';
                sentPacket0.style.transform = 'translateX(-25px)';
                sentPacket0.style.transition = 'all 0.5s ease-in-out';
                
                setTimeout(() => {
                    sentPacket0.style.opacity = '1';
                    sentPacket0.style.transform = 'translateX(80px)';
                    sentPacket0.style.transition = 'all 1s ease-in-out';
                }, 500);
            }, 500);
            
            setTimeout(() => {
                const sentPacket1 = document.getElementById('sent-packet-1');
                sentPacket1.innerText = 'Pkt 4';
                sentPacket1.style.opacity = '0';
                sentPacket1.style.transform = 'translateX(-25px)';
                sentPacket1.style.transition = 'all 0.5s ease-in-out';
                
                setTimeout(() => {
                    sentPacket1.style.opacity = '1';
                    sentPacket1.style.transform = 'translateX(80px)';
                    sentPacket1.style.transition = 'all 1s ease-in-out';
                }, 500);
            }, 1000);
            
            setTimeout(() => {
                const sentPacket2 = document.getElementById('sent-packet-2');
                sentPacket2.innerText = 'Pkt 5';
                sentPacket2.style.opacity = '0';
                sentPacket2.style.transform = 'translateX(-25px)';
                sentPacket2.style.transition = 'all 0.5s ease-in-out';
                
                setTimeout(() => {
                    sentPacket2.style.opacity = '1';
                    sentPacket2.style.transform = 'translateX(80px)';
                    sentPacket2.style.transition = 'all 1s ease-in-out';
                }, 500);
            }, 1500);
        }, 6000));
        
        // 步骤4: 接收方接收并确认数据包3-5
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '接收方接收数据包3-5并发送确认ACK 6';
            senderDevice.style.borderColor = '#3498db';
            senderDevice.style.boxShadow = 'none';
            receiverDevice.style.borderColor = '#ff9800';
            receiverDevice.style.boxShadow = '0 0 10px rgba(255, 193, 7, 0.5)';
            
            // 更新接收窗口的数据包状态
            for (let i = 3; i < 6; i++) {
                const receiverPacket = document.getElementById(`receiver-packet-${i}`);
                receiverPacket.style.backgroundColor = 'rgba(46, 204, 113, 0.9)';
                receiverPacket.style.borderColor = '#27ae60';
                receiverPacket.style.color = 'white';
            }
            
            // 发送确认包
            const ackPacket = document.getElementById('ack-packet');
            ackPacket.innerText = 'ACK 6';
            ackPacket.style.opacity = '0';
            ackPacket.style.transform = 'translateX(80px)';
            ackPacket.style.transition = 'all 0.5s ease-in-out';
            
            setTimeout(() => {
                ackPacket.style.opacity = '1';
                ackPacket.style.transform = 'translateX(-80px)';
                ackPacket.style.transition = 'all 1s ease-in-out';
            }, 500);
        }, 10000));
        
        // 步骤5: 发送方滑动窗口，发送数据包6-8
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '发送方滑动窗口，发送新的数据包6-8';
            receiverDevice.style.borderColor = '#e74c3c';
            receiverDevice.style.boxShadow = 'none';
            senderDevice.style.borderColor = '#27ae60';
            senderDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            
            // 滑动发送窗口
            senderWindowBorder.style.left = `${9 * 20}px`;
            senderWindowBorder.style.width = `${1 * 20}px`;
            senderWindowBorder.style.transition = 'all 0.5s ease-in-out';
            
            // 更新发送窗口的数据包状态
            for (let i = 3; i < 6; i++) {
                const senderPacket = document.getElementById(`sender-packet-${i}`);
                senderPacket.style.backgroundColor = 'rgba(52, 152, 219, 0.9)';
                senderPacket.style.borderColor = '#2980b9';
            }
            
            for (let i = 6; i < 9; i++) {
                const senderPacket = document.getElementById(`sender-packet-${i}`);
                senderPacket.style.backgroundColor = 'rgba(46, 204, 113, 0.9)';
                senderPacket.style.borderColor = '#27ae60';
                senderPacket.style.color = 'white';
            }
            
            // 发送新的数据包
            setTimeout(() => {
                const sentPacket0 = document.getElementById('sent-packet-0');
                sentPacket0.innerText = 'Pkt 6';
                sentPacket0.style.opacity = '0';
                sentPacket0.style.transform = 'translateX(-25px)';
                sentPacket0.style.transition = 'all 0.5s ease-in-out';
                
                setTimeout(() => {
                    sentPacket0.style.opacity = '1';
                    sentPacket0.style.transform = 'translateX(80px)';
                    sentPacket0.style.transition = 'all 1s ease-in-out';
                }, 500);
            }, 500);
            
            setTimeout(() => {
                const sentPacket1 = document.getElementById('sent-packet-1');
                sentPacket1.innerText = 'Pkt 7';
                sentPacket1.style.opacity = '0';
                sentPacket1.style.transform = 'translateX(-25px)';
                sentPacket1.style.transition = 'all 0.5s ease-in-out';
                
                setTimeout(() => {
                    sentPacket1.style.opacity = '1';
                    sentPacket1.style.transform = 'translateX(80px)';
                    sentPacket1.style.transition = 'all 1s ease-in-out';
                }, 500);
            }, 1000);
            
            setTimeout(() => {
                const sentPacket2 = document.getElementById('sent-packet-2');
                sentPacket2.innerText = 'Pkt 8';
                sentPacket2.style.opacity = '0';
                sentPacket2.style.transform = 'translateX(-25px)';
                sentPacket2.style.transition = 'all 0.5s ease-in-out';
                
                setTimeout(() => {
                    sentPacket2.style.opacity = '1';
                    sentPacket2.style.transform = 'translateX(80px)';
                    sentPacket2.style.transition = 'all 1s ease-in-out';
                }, 500);
            }, 1500);
        }, 13000));
        
        // 步骤6: 接收方接收并确认数据包6-8
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '接收方接收数据包6-8并发送确认ACK 9';
            senderDevice.style.borderColor = '#3498db';
            senderDevice.style.boxShadow = 'none';
            receiverDevice.style.borderColor = '#ff9800';
            receiverDevice.style.boxShadow = '0 0 10px rgba(255, 193, 7, 0.5)';
            
            // 更新接收窗口的数据包状态
            for (let i = 6; i < 9; i++) {
                const receiverPacket = document.getElementById(`receiver-packet-${i}`);
                receiverPacket.style.backgroundColor = 'rgba(46, 204, 113, 0.9)';
                receiverPacket.style.borderColor = '#27ae60';
                receiverPacket.style.color = 'white';
            }
            
            // 发送确认包
            const ackPacket = document.getElementById('ack-packet');
            ackPacket.innerText = 'ACK 9';
            ackPacket.style.opacity = '0';
            ackPacket.style.transform = 'translateX(80px)';
            ackPacket.style.transition = 'all 0.5s ease-in-out';
            
            setTimeout(() => {
                ackPacket.style.opacity = '1';
                ackPacket.style.transform = 'translateX(-80px)';
                ackPacket.style.transition = 'all 1s ease-in-out';
            }, 500);
        }, 17000));
        
        // 步骤7: 发送方滑动窗口，发送最后一个数据包9
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '发送方滑动窗口，发送最后一个数据包9';
            receiverDevice.style.borderColor = '#e74c3c';
            receiverDevice.style.boxShadow = 'none';
            senderDevice.style.borderColor = '#27ae60';
            senderDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            
            // 更新发送窗口的数据包状态
            for (let i = 6; i < 9; i++) {
                const senderPacket = document.getElementById(`sender-packet-${i}`);
                senderPacket.style.backgroundColor = 'rgba(52, 152, 219, 0.9)';
                senderPacket.style.borderColor = '#2980b9';
            }
            
            const senderPacket9 = document.getElementById('sender-packet-9');
            if (senderPacket9) {
                senderPacket9.style.backgroundColor = 'rgba(46, 204, 113, 0.9)';
                senderPacket9.style.borderColor = '#27ae60';
                senderPacket9.style.color = 'white';
            }
            
            // 发送最后一个数据包
            setTimeout(() => {
                const sentPacket0 = document.getElementById('sent-packet-0');
                sentPacket0.innerText = 'Pkt 9';
                sentPacket0.style.opacity = '0';
                sentPacket0.style.transform = 'translateX(-25px)';
                sentPacket0.style.transition = 'all 0.5s ease-in-out';
                
                setTimeout(() => {
                    sentPacket0.style.opacity = '1';
                    sentPacket0.style.transform = 'translateX(80px)';
                    sentPacket0.style.transition = 'all 1s ease-in-out';
                }, 500);
            }, 500);
        }, 20000));
        
        // 步骤8: 接收方接收并确认最后一个数据包9
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '接收方接收最后一个数据包9并发送确认ACK 10';
            senderDevice.style.borderColor = '#3498db';
            senderDevice.style.boxShadow = 'none';
            receiverDevice.style.borderColor = '#ff9800';
            receiverDevice.style.boxShadow = '0 0 10px rgba(255, 193, 7, 0.5)';
            
            // 更新接收窗口的数据包状态
            const receiverPacket9 = document.getElementById('receiver-packet-9');
            if (receiverPacket9) {
                receiverPacket9.style.backgroundColor = 'rgba(46, 204, 113, 0.9)';
                receiverPacket9.style.borderColor = '#27ae60';
                receiverPacket9.style.color = 'white';
            }
            
            // 发送确认包
            const ackPacket = document.getElementById('ack-packet');
            ackPacket.innerText = 'ACK 10';
            ackPacket.style.opacity = '0';
            ackPacket.style.transform = 'translateX(80px)';
            ackPacket.style.transition = 'all 0.5s ease-in-out';
            
            setTimeout(() => {
                ackPacket.style.opacity = '1';
                ackPacket.style.transform = 'translateX(-80px)';
                ackPacket.style.transition = 'all 1s ease-in-out';
            }, 500);
        }, 23000));
        
        // 完成状态
        principleTimers.push(setTimeout(() => {
            status.innerHTML = '所有数据包(0-9)传输完成！滑动窗口机制演示结束。';
            status.style.backgroundColor = '#27ae60';
            status.style.color = 'white';
            
            // 让所有设备边框高亮
            senderDevice.style.borderColor = '#27ae60';
            receiverDevice.style.borderColor = '#27ae60';
            senderDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            receiverDevice.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
            
            // 动画结束，启用开始按钮
            const startBtn = document.getElementById('start-window-btn');
            if (startBtn) {
                startBtn.disabled = false;
                startBtn.style.opacity = '1';
                startBtn.style.cursor = 'pointer';
            }
            isAnimating = false;
        }, 27000));
    }
    
    // 原理展示功能
    function showPrinciple(principleId) {
        console.log('显示原理展示:', principleId);
        
        // 清除所有之前设置的定时器，防止动画冲突
        principleTimers.forEach(timer => clearTimeout(timer));
        principleTimers = [];
        
        // 设置动画标志为true
        isAnimating = true;
        
        // 根据原理ID显示不同的原理内容
        switch (principleId) {
            case 'layered-model':
                showLayeredModel();
                break;
            case 'data-packaging':
                // 数据分包原理展示
                showDataPackaging();
                break;
            case 'three-way-handshake':
                // 三次握手原理展示
                showThreeWayHandshake();
                break;
            case 'digital-signature':
                // 数字签名原理展示
                showDigitalSignature();
                break;
            case 'secure-web':
                // 安全WEB原理展示
                showSecureWeb();
                break;
            case 'sliding-window':
                // 滑动窗口原理展示
                showSlidingWindow();
                break;
            default:
                alert(`准备展示${principleId}原理...`);
        }
    }
    
    // 显示数据分包原理（IP分片处理）
    function showDataPackaging() {
        const modal = document.getElementById('modal');
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <div class="modal-header">
                <h3>数据分包原理 - IP分片处理</h3>
            </div>
            <div class="property-form">
                <!-- 动画展示区域 -->
                <div class="form-group">
                    <label>IP分片与重组过程</label>
                    <div id="data-packaging-animation" style="
                        background-color: #f8f9fa;
                        padding: 15px;
                        border-radius: 8px;
                        margin-top: 10px;
                        min-height: 400px;
                        max-height: 500px;
                        overflow-y: auto;
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        width: 100%;
                    ">
                        <!-- 发送端、中间方和接收方 -->
                        <div style="
                            display: flex;
                            gap: 30px;
                            align-items: flex-start;
                            position: relative;
                            width: 100%;
                            justify-content: center;
                            margin-bottom: 20px;
                        ">
                            <!-- 发送端 -->
                            <div style="
                                display: flex;
                                flex-direction: column;
                                align-items: center;
                                width: 18%;
                            ">
                                <h5 style="margin-bottom: 10px; font-size: 14px; text-align: center;">发送端</h5>
                                <div id="sender-device" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #3498db;
                                    border-radius: 8px;
                                    padding: 15px;
                                    width: 100%;
                                    min-height: 120px;
                                    display: flex;
                                    flex-direction: column;
                                    align-items: center;
                                    justify-content: center;
                                ">
                                    <div style="font-size: 40px; margin-bottom: 5px;">💻</div>
                                    <div style="font-size: 12px;">PC1</div>
                                    <div style="font-size: 10px; margin-top: 5px;">192.168.1.100</div>
                                </div>
                                <!-- 发送端原始数据包 -->
                                <div id="sender-content" style="
                                    margin-top: 10px;
                                    width: 100%;
                                    text-align: center;
                                ">
                                    <!-- 原始数据包 -->
                                    <div id="original-packet" style="
                                        background-color: rgba(52, 152, 219, 0.9);
                                        color: white;
                                        padding: 8px 12px;
                                        border-radius: 4px;
                                        font-size: 12px;
                                        text-align: center;
                                        border: 2px solid #2980b9;
                                    ">
                                        原始数据包<br/>
                                        <span style="font-size: 10px;">(5000字节)</span>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- 中间方区域 -->
                            <div style="
                                display: flex;
                                flex-direction: column;
                                align-items: center;
                                width: 65%;
                            ">
                                <!-- 网络连接 -->
                                <div style="
                                    width: 100%;
                                    height: 100px;
                                    margin: 0;
                                    display: flex;
                                    justify-content: space-between;
                                    position: relative;
                                ">
                                    <div style="
                                        width: 100%;
                                        height: 4px;
                                        background-color: #bdc3c7;
                                        position: absolute;
                                        top: 60px;
                                        left: 0;
                                    "></div>
                                    
                                    <!-- 中间方路由器 -->
                                    <div style="
                                        z-index: 5;
                                        text-align: center;
                                        display: flex;
                                        flex-direction: column;
                                        align-items: center;
                                        justify-content: flex-start;
                                        position: absolute;
                                        bottom: 0px;
                                        left: 15%;
                                    ">
                                        <div style="width: 50px; height: 50px; margin-bottom: 3px;">
                                            <svg width='50' height='50' viewBox='0 0 32 32' xmlns='http://www.w3.org/2000/svg'><rect width='28' height='16' x='2' y='8' rx='2' fill='#5c6bc0'/><rect width='26' height='12' x='3' y='10' rx='1' fill='#3f51b5'/><rect width='2' height='4' x='5' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='8' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='11' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='14' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='17' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='20' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='23' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='26' y='14' rx='1' fill='#7986cb'/><circle cx='7' cy='12' r='1' fill='#4caf50'/><circle cx='13' cy='12' r='1' fill='#ffeb3b'/><circle cx='19' cy='12' r='1' fill='#ff9800'/><circle cx='25' cy='12' r='1' fill='#f44336'/><rect width='22' height='2' x='5' y='22' rx='1' fill='#9fa8da'/></svg>
                                        </div>
                                        <div style="font-size: 13px; font-weight: bold; color: #333; margin-top: -5px;">R1</div>
                                    </div>
                                    
                                    <div style="
                                        z-index: 5;
                                        text-align: center;
                                        display: flex;
                                        flex-direction: column;
                                        align-items: center;
                                        justify-content: flex-start;
                                        position: absolute;
                                        bottom: 0px;
                                        left: 50%;
                                        transform: translateX(-50%);
                                    ">
                                        <div style="width: 50px; height: 50px; margin-bottom: 3px;">
                                            <svg width='50' height='50' viewBox='0 0 32 32' xmlns='http://www.w3.org/2000/svg'><rect width='28' height='16' x='2' y='8' rx='2' fill='#5c6bc0'/><rect width='26' height='12' x='3' y='10' rx='1' fill='#3f51b5'/><rect width='2' height='4' x='5' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='8' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='11' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='14' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='17' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='20' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='23' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='26' y='14' rx='1' fill='#7986cb'/><circle cx='7' cy='12' r='1' fill='#4caf50'/><circle cx='13' cy='12' r='1' fill='#ffeb3b'/><circle cx='19' cy='12' r='1' fill='#ff9800'/><circle cx='25' cy='12' r='1' fill='#f44336'/><rect width='22' height='2' x='5' y='22' rx='1' fill='#9fa8da'/></svg>
                                        </div>
                                        <div style="font-size: 13px; font-weight: bold; color: #333; margin-top: -5px;">R2</div>
                                    </div>
                                    
                                    <div style="
                                        z-index: 5;
                                        text-align: center;
                                        display: flex;
                                        flex-direction: column;
                                        align-items: center;
                                        justify-content: flex-start;
                                        position: absolute;
                                        bottom: 0px;
                                        right: 15%;
                                    ">
                                        <div style="width: 50px; height: 50px; margin-bottom: 3px;">
                                            <svg width='50' height='50' viewBox='0 0 32 32' xmlns='http://www.w3.org/2000/svg'><rect width='28' height='16' x='2' y='8' rx='2' fill='#5c6bc0'/><rect width='26' height='12' x='3' y='10' rx='1' fill='#3f51b5'/><rect width='2' height='4' x='5' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='8' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='11' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='14' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='17' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='20' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='23' y='14' rx='1' fill='#7986cb'/><rect width='2' height='4' x='26' y='14' rx='1' fill='#7986cb'/><circle cx='7' cy='12' r='1' fill='#4caf50'/><circle cx='13' cy='12' r='1' fill='#ffeb3b'/><circle cx='19' cy='12' r='1' fill='#ff9800'/><circle cx='25' cy='12' r='1' fill='#f44336'/><rect width='22' height='2' x='5' y='22' rx='1' fill='#9fa8da'/></svg>
                                        </div>
                                        <div style="font-size: 13px; font-weight: bold; color: #333; margin-top: -5px;">R3</div>
                                    </div>
                                    
                                    <!-- 数据流动效果容器 -->
                                    <div id="data-flow-container" style="
                                        position: absolute;
                                        top: 0;
                                        left: 0;
                                        width: 100%;
                                        height: 100%;
                                        pointer-events: none;
                                        z-index: 10;
                                    "></div>
                                </div>
                                
                                <!-- 分片展示区域 -->
                                <div id="fragmentation-area" style="
                                    background-color: white;
                                    border: 1px solid #ddd;
                                    border-radius: 8px;
                                    padding: 15px;
                                    width: 100%;
                                    min-height: 200px;
                                    display: flex;
                                    flex-direction: column;
                                    align-items: center;
                                    justify-content: space-around;
                                ">
                                    <!-- 分片将在这里动态生成 -->
                                    <div style="font-size: 13px; color: #7f8c8d; margin-bottom: 10px;">
                                        IP分片过程中...
                                    </div>
                                    <div id="fragments-container" style="
                                        display: flex;
                                        justify-content: center;
                                        width: 100%;
                                        flex-direction: column;
                                        align-items: center;
                                        gap: 10px;
                                    "></div>
                                </div>
                            </div>
                            
                            <!-- 接收方 -->
                            <div style="
                                display: flex;
                                flex-direction: column;
                                align-items: center;
                                width: 18%;
                            ">
                                <h5 style="margin-bottom: 10px; font-size: 14px; text-align: center;">接收端</h5>
                                <div id="receiver-device" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #3498db;
                                    border-radius: 8px;
                                    padding: 15px;
                                    width: 100%;
                                    min-height: 120px;
                                    display: flex;
                                    flex-direction: column;
                                    align-items: center;
                                    justify-content: center;
                                ">
                                    <div style="font-size: 40px; margin-bottom: 5px;">💻</div>
                                    <div style="font-size: 12px;">PC2</div>
                                    <div style="font-size: 10px; margin-top: 5px;">192.168.2.100</div>
                                </div>
                                <!-- 接收端重组数据包 -->
                                <div id="receiver-content" style="
                                    margin-top: 10px;
                                    width: 100%;
                                    text-align: center;
                                ">

                                    
                                    <!-- 重组数据包 -->
                                    <div id="reassembled-packet" style="
                                        background-color: rgba(46, 204, 113, 0.9);
                                        color: white;
                                        padding: 8px 12px;
                                        border-radius: 4px;
                                        font-size: 10px;
                                        text-align: center;
                                        border: 2px solid #27ae60;
                                        display: none;
                                    ">
                                        重组数据包<br/>
                                        <span style="font-size: 8px;">(5000字节)</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 控制按钮 -->
                    </div>
                </div>
                
                <!-- 控制按钮 -->
                <div class="form-actions" style="margin-top: 15px; display: flex; gap: 10px; justify-content: center;">
                    <button id="start-fragmentation-btn" class="btn-confirm">开始</button>
                    <button id="close-fragmentation-btn" class="btn-cancel">关闭</button>
                </div>
            </div>
        `;
        
        modal.style.display = 'block';
        initModalDrag();
        
        // 添加事件监听器
        const startBtn = document.getElementById('start-fragmentation-btn');
        const closeBtn = document.getElementById('close-fragmentation-btn');
        
        if (startBtn) {
            startBtn.addEventListener('click', startFragmentationAnimation);
        }
        
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                document.getElementById('modal').style.display = 'none';
            });
        }
    }
    
    // IP分片动画函数
    window.startFragmentationAnimation = function() {
        console.log('开始IP分片动画');
        
        // 禁用开始按钮
        const startBtn = document.getElementById('start-fragmentation-btn');
        if (startBtn) {
            startBtn.disabled = true;
            startBtn.style.opacity = '0.5';
            startBtn.style.cursor = 'not-allowed';
        }
        
        // 定义分片信息
        const fragments = [
            {
                id: 1,
                totalLength: 1500, // 包括IP头的总长度
                dataLength: 1480,  // 数据部分长度
                offset: 0,         // 偏移量（以8字节为单位）
                moreFragments: 1   // 还有更多分片
            },
            {
                id: 2,
                totalLength: 1500,
                dataLength: 1480,
                offset: 185,      // 1480 / 8 = 185
                moreFragments: 1
            },
            {
                id: 3,
                totalLength: 1500,
                dataLength: 1480,
                offset: 370,      // 2960 / 8 = 370
                moreFragments: 1
            },
            {
                id: 4,
                totalLength: 520,
                dataLength: 500,  // 5000 - 1480*3 = 560，但用户要求500字节
                offset: 555,      // 4440 / 8 = 555
                moreFragments: 0   // 最后一个分片
            }
        ];
        
        // 显示分片
        const fragmentsContainer = document.getElementById('fragments-container');
        const fragmentationArea = document.getElementById('fragmentation-area');
        
        // 更新状态文本 - 使用更精确的选择器
        const statusText = fragmentationArea.querySelector('div:first-child');
        if (statusText) {
            console.log('更新状态文本:', statusText);
            statusText.textContent = 'IP分片结果:';
            statusText.style.color = '#3498db';
            statusText.style.fontWeight = 'bold';
        } else {
            console.error('未找到状态文本元素');
        }
        
        // 清空容器
        fragmentsContainer.innerHTML = '';
        
        // 创建并添加分片元素
        fragments.forEach((fragment, index) => {
            // 添加延迟，使分片依次出现
            setTimeout(() => {
                const fragmentElement = createFragmentElement(fragment);
                fragmentsContainer.appendChild(fragmentElement);
                
                // 添加淡入动画
                fragmentElement.style.opacity = '0';
                fragmentElement.style.transform = 'scale(0.8)';
                setTimeout(() => {
                    fragmentElement.style.transition = 'opacity 0.5s, transform 0.5s';
                    fragmentElement.style.opacity = '1';
                    fragmentElement.style.transform = 'scale(1)';
                }, 50);
                
                // 最后一个分片显示后，开始传输动画
                if (index === fragments.length - 1) {
                    setTimeout(() => {
                        animateFragmentsTransmission(fragments);
                    }, 500);
                }
            }, index * 800);
        });
    }
    
    // 创建分片元素
    function createFragmentElement(fragment) {
        const fragmentElement = document.createElement('div');
        fragmentElement.style.cssText = `
            background-color: rgba(52, 152, 219, 0.1);
            border: 2px solid #3498db;
            border-radius: 8px;
            padding: 12px;
            width: 60%;
            min-height: 100px;
            display: flex;
            flex-direction: column;
            align-items: center;
            min-width: 250px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        `;
        
        fragmentElement.innerHTML = `
            <div style="font-weight: bold; color: #3498db; margin-bottom: 8px; font-size: 12px;">
                IP分片 #${fragment.id}
            </div>
            <div style="font-size: 10px; margin-bottom: 4px;">
                总长度: ${fragment.totalLength} 字节
            </div>
            <div style="font-size: 10px; margin-bottom: 4px;">
                数据长度: ${fragment.dataLength} 字节
            </div>
            <div style="font-size: 10px; margin-bottom: 4px;">
                偏移量: ${fragment.offset} (${fragment.offset * 8} 字节)
            </div>
            <div style="font-size: 10px; margin-bottom: 4px;">
                更多分片: ${fragment.moreFragments ? '是' : '否'}
            </div>
            <div style="
                background-color: #3498db;
                color: white;
                padding: 4px 8px;
                border-radius: 4px;
                font-size: 8px;
                margin-top: 8px;
            ">
                正在传输...
            </div>
        `;
        
        return fragmentElement;
    }
    
    // 分片传输动画
    function animateFragmentsTransmission(fragments) {
        const fragmentsContainer = document.getElementById('fragments-container');
        const fragmentElements = fragmentsContainer.querySelectorAll('div');
        const fragmentationArea = document.getElementById('fragmentation-area');
        
        // 更新状态文本
        const statusText = fragmentationArea.querySelector('div:first-child');
        if (statusText) {
            statusText.textContent = '分片正在网络中传输:';
            statusText.style.color = '#f39c12';
        }
        
        // 添加数据流动效果
        createDataFlowEffect();
        
        // 模拟分片在网络中的传输
        fragmentElements.forEach((element, index) => {
            // 设置初始状态
            element.style.transition = 'opacity 0.5s, background-color 0.5s';
            
            // 高亮显示当前传输的分片
            element.style.backgroundColor = 'rgba(243, 156, 18, 0.2)';
            
            // 更新状态为传输中
            const statusDiv = element.querySelector('div:last-child');
            if (statusDiv) {
                statusDiv.textContent = '正在传输...';
                statusDiv.style.backgroundColor = '#f39c12';
            }
            
            // 添加简单的闪烁动画效果
            let blinkCount = 0;
            const blinkInterval = setInterval(() => {
                element.style.opacity = blinkCount % 2 === 0 ? '1' : '0.7';
                blinkCount++;
                if (blinkCount >= 6) {
                    clearInterval(blinkInterval);
                    element.style.opacity = '1';
                    
                    // 更新状态为已到达
                    setTimeout(() => {
                        if (statusDiv) {
                            statusDiv.textContent = '已到达中间节点';
                            statusDiv.style.backgroundColor = '#27ae60';
                        }
                        element.style.backgroundColor = 'rgba(52, 152, 219, 0.1)';
                    }, 500);
                }
            }, 300);
        });
        
        // 所有分片传输完成后，开始重组
        setTimeout(() => {
            reassembleFragments();
        }, fragments.length * 2000 + 1000);
    }
    
    // 创建数据流动效果
    function createDataFlowEffect() {
        const dataFlowContainer = document.getElementById('data-flow-container');
        if (!dataFlowContainer) return;
        
        // 清除之前的数据流动效果
        dataFlowContainer.innerHTML = '';
        
        // 创建3个数据流动动画，模拟数据包在路由器之间传输
        for (let i = 0; i < 3; i++) {
            setTimeout(() => {
                const dataPacket = document.createElement('div');
                dataPacket.style.cssText = `
                    position: absolute;
                    width: 10px;
                    height: 10px;
                    background-color: #f39c12;
                    border-radius: 50%;
                    box-shadow: 0 0 10px #f39c12;
                    left: 100px;
                    top: 60px;
                    transform: translateY(-50%);
                    opacity: 0;
                    z-index: 100;
                `;
                
                dataFlowContainer.appendChild(dataPacket);
                
                // 动画：从发送方到接收方
                setTimeout(() => {
                    dataPacket.style.transition = 'left 3s linear, opacity 0.5s';
                    dataPacket.style.opacity = '1';
                    dataPacket.style.left = 'calc(100% - 100px)';
                }, 50);
                
                // 动画结束后移除元素
                setTimeout(() => {
                    dataPacket.remove();
                }, 3500);
            }, i * 1000);
        }
    }
    
    // 重组分片动画
    function reassembleFragments() {
        console.log('开始重组分片');
        
        // 确保获取到所有元素
        const fragmentsContainer = document.getElementById('fragments-container');
        const fragmentElements = fragmentsContainer.querySelectorAll('div');
        const fragmentationArea = document.getElementById('fragmentation-area');
        const reassembledPacket = document.getElementById('reassembled-packet');
        
        if (!reassembledPacket) {
            console.error('reassembled-packet element not found');
            return;
        }
        
        // 更新传输区域状态 - 使用精确选择器
        const statusText = fragmentationArea.querySelector('div:first-child');
        if (statusText) {
            statusText.textContent = '接收方正在重组分片:';
            statusText.style.color = '#27ae60';
        }
        
        // 更新分片状态为已到达接收方
        fragmentElements.forEach((element, index) => {
            setTimeout(() => {
                const statusDiv = element.querySelector('div:last-child');
                if (statusDiv) {
                    statusDiv.textContent = '已到达接收方';
                    statusDiv.style.backgroundColor = '#2ecc71';
                }
            }, index * 300);
        });
        
        // 确保接收方的重组数据包显示
        reassembledPacket.style.display = 'block';
        reassembledPacket.style.opacity = '1';
        reassembledPacket.style.transform = 'scale(1)';
        
        // 更新传输区域状态 - 使用精确选择器
        setTimeout(() => {
            const statusText = fragmentationArea.querySelector('div:first-child');
            if (statusText) {
                statusText.textContent = '数据包重组完成!';
                statusText.style.color = '#27ae60';
                statusText.style.fontWeight = 'bold';
            }
            
            // 启用开始按钮
            const startBtn = document.getElementById('start-fragmentation-btn');
            if (startBtn) {
                startBtn.disabled = false;
                startBtn.style.opacity = '1';
                startBtn.style.cursor = 'pointer';
            }
        }, 1000);
    }
    
    // 重置分片动画
    window.resetFragmentationAnimation = function() {
        console.log('重置IP分片动画');
        
        // 启用开始按钮
        const startBtn = document.getElementById('start-fragmentation-btn');
        if (startBtn) {
            startBtn.disabled = false;
            startBtn.style.opacity = '1';
            startBtn.style.cursor = 'pointer';
        }
        
        // 清空分片容器
        const fragmentsContainer = document.getElementById('fragments-container');
        fragmentsContainer.innerHTML = '';
        
        // 重置传输区域状态 - 使用精确选择器
        const fragmentationArea = document.getElementById('fragmentation-area');
        const statusText = fragmentationArea.querySelector('div:first-child');
        if (statusText) {
            statusText.textContent = 'IP分片过程中...';
            statusText.style.color = '#7f8c8d';
            statusText.style.fontWeight = 'normal';
        }
        
        // 隐藏重组后的数据包
        const reassembledPacket = document.getElementById('reassembled-packet');
        reassembledPacket.style.display = 'none';
    }
    
    // 显示数字签名原理
    function showDigitalSignature() {
        const modal = document.getElementById('modal');
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <div class="modal-header">
                <h3>数字签名原理 - 信息安全验证</h3>
            </div>
            <div class="property-form">
                <!-- 动画展示区域 -->
                <div class="form-group">
                    <label>数字签名过程 - 确保信息完整性与不可否认性</label>
                    <div id="digital-signature-animation" style="
                        background-color: #f8f9fa;
                        padding: 15px;
                        border-radius: 8px;
                        margin-top: 10px;
                        min-height: 400px;
                        max-height: 500px;
                        overflow: hidden;
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        width: 100%;
                        max-width: 550px;
                        margin-left: auto;
                        margin-right: auto;
                    ">
                        <!-- 发送端和接收端 -->
                        <div style="
                            display: flex;
                            gap: 30px;
                            align-items: flex-start;
                            position: relative;
                            width: 100%;
                            justify-content: center;
                        ">
                            <!-- 发送端 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">发送端</h5>
                                <div id="sender-digital-signature" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #3498db;
                                    border-radius: 8px;
                                    padding: 10px;
                                    min-height: 180px;
                                    width: 180px;
                                    display: flex;
                                    flex-direction: column;
                                    gap: 8px;
                                ">
                                    <!-- 消息 -->
                                    <div id="sender-message" style="
                                        background-color: white;
                                        border: 1px solid #ddd;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 11px;
                                    ">
                                        <strong>消息:</strong><br/>
                                        重要消息
                                    </div>
                                    <!-- 哈希值 -->
                                    <div id="sender-hash" style="
                                        background-color: rgba(46, 204, 113, 0.1);
                                        border: 1px solid #27ae60;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                        display: none;
                                    ">
                                        <strong>哈希值:</strong><br/>
                                        0x1A2B3C4D...
                                    </div>
                                    <!-- 私钥 -->
                                    <div id="sender-private-key" style="
                                        background-color: rgba(231, 76, 60, 0.1);
                                        border: 1px solid #e74c3c;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                    ">
                                        <strong>私钥:</strong><br/>
                                        [发送端私钥]
                                    </div>
                                    <!-- 数字签名 -->
                                    <div id="digital-signature" style="
                                        background-color: rgba(155, 89, 182, 0.1);
                                        border: 1px solid #9b59b6;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                        display: none;
                                    ">
                                        <strong>数字签名:</strong><br/>
                                        [签名内容]
                                    </div>
                                </div>
                            </div>
                            
                            <!-- 传输路径 -->
                            <div style="
                                position: relative;
                                width: 100px;
                                height: 100px;
                                display: flex;
                                align-items: center;
                                justify-content: center;
                            ">
                                <div style="
                                    width: 100%;
                                    height: 4px;
                                    background-color: #bdc3c7;
                                    position: absolute;
                                    top: 50%;
                                    left: 0;
                                "></div>
                                <!-- 数据流动效果容器 -->
                                <div id="signature-flow-container" style="
                                    position: absolute;
                                    top: 0;
                                    left: 0;
                                    width: 100%;
                                    height: 100%;
                                    pointer-events: none;
                                    z-index: 10;
                                "></div>
                            </div>
                            
                            <!-- 接收端 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">接收端</h5>
                                <div id="receiver-digital-signature" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #3498db;
                                    border-radius: 8px;
                                    padding: 10px;
                                    min-height: 180px;
                                    width: 180px;
                                    display: flex;
                                    flex-direction: column;
                                    gap: 8px;
                                ">
                                    <!-- 接收的消息 -->
                                    <div id="receiver-message" style="
                                        background-color: white;
                                        border: 1px solid #ddd;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 11px;
                                        display: none;
                                    ">
                                        <strong>接收的消息:</strong><br/>
                                        重要消息
                                    </div>
                                    <!-- 接收的签名 -->
                                    <div id="receiver-signature" style="
                                        background-color: rgba(155, 89, 182, 0.1);
                                        border: 1px solid #9b59b6;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                        display: none;
                                    ">
                                        <strong>接收的签名:</strong><br/>
                                        [签名内容]
                                    </div>
                                    <!-- 公钥 -->
                                    <div id="receiver-public-key" style="
                                        background-color: rgba(52, 152, 219, 0.1);
                                        border: 1px solid #3498db;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                    ">
                                        <strong>公钥:</strong><br/>
                                        [发送端公钥]
                                    </div>
                                    <!-- 验证结果 -->
                                    <div id="verification-result" style="
                                        background-color: rgba(231, 76, 60, 0.1);
                                        border: 1px solid #e74c3c;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 11px;
                                        text-align: center;
                                        display: none;
                                    ">
                                        验证中...
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- 控制按钮 -->
                    <div class="form-actions" style="margin-top: 15px; display: flex; gap: 10px; justify-content: center;">
                        <button id="start-signature-btn" class="btn-confirm">开始</button>
                        <button id="close-signature-btn" class="btn-cancel">关闭</button>
                    </div>
                </div>
            </div>
        `;
        
        modal.style.display = 'block';
        initModalDrag();
        
        // 添加事件监听器
        const startBtn = document.getElementById('start-signature-btn');
        const closeBtn = document.getElementById('close-signature-btn');
        
        if (startBtn) {
            startBtn.addEventListener('click', startDigitalSignatureAnimation);
        }
        
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                document.getElementById('modal').style.display = 'none';
            });
        }
    }
    
    // 数字签名动画函数
    window.startDigitalSignatureAnimation = function() {
        console.log('开始数字签名动画');
        
        // 禁用开始按钮
        const startBtn = document.getElementById('start-signature-btn');
        if (startBtn) {
            startBtn.disabled = true;
            startBtn.style.opacity = '0.5';
            startBtn.style.cursor = 'not-allowed';
        }
        
        const senderMessage = document.getElementById('sender-message');
        const senderHash = document.getElementById('sender-hash');
        const senderPrivateKey = document.getElementById('sender-private-key');
        const digitalSignature = document.getElementById('digital-signature');
        const receiverMessage = document.getElementById('receiver-message');
        const receiverSignature = document.getElementById('receiver-signature');
        const receiverPublicKey = document.getElementById('receiver-public-key');
        const verificationResult = document.getElementById('verification-result');
        const flowContainer = document.getElementById('signature-flow-container');
        
        // 1. 高亮显示原始消息 (500ms)
        setTimeout(() => {
            senderMessage.style.backgroundColor = 'rgba(52, 152, 219, 0.1)';
            senderMessage.style.borderColor = '#3498db';
        }, 500);
        
        // 2. 生成哈希值 (1000ms)
        setTimeout(() => {
            senderHash.style.display = 'block';
            senderHash.style.opacity = '1';
            senderHash.style.transition = 'opacity 0.5s';
            
            // 添加哈希计算提示
            const hashTip = document.createElement('div');
            hashTip.textContent = '计算哈希值...';
            hashTip.style.cssText = `
                position: absolute;
                top: 50%;
                left: 10px;
                font-size: 8px;
                color: #27ae60;
                animation: blink 1s infinite;
            `;
            senderHash.appendChild(hashTip);
        }, 1000);
        
        // 3. 准备使用私钥 (1500ms)
        setTimeout(() => {
            senderPrivateKey.style.backgroundColor = 'rgba(231, 76, 60, 0.2)';
            senderPrivateKey.style.borderColor = '#e74c3c';
            
            // 移除哈希提示
            const hashTip = senderHash.querySelector('div');
            if (hashTip) hashTip.remove();
        }, 1500);
        
        // 4. 生成数字签名 (2000ms)
        setTimeout(() => {
            digitalSignature.style.display = 'block';
            digitalSignature.style.opacity = '1';
            digitalSignature.style.transition = 'opacity 0.5s';
            
            // 添加签名生成提示
            const sigTip = document.createElement('div');
            sigTip.textContent = '生成签名...';
            sigTip.style.cssText = `
                position: absolute;
                top: 50%;
                left: 10px;
                font-size: 8px;
                color: #9b59b6;
                animation: blink 1s infinite;
            `;
            digitalSignature.appendChild(sigTip);
        }, 2000);
        
        // 5. 重置发送端高亮 (2500ms)
        setTimeout(() => {
            senderMessage.style.backgroundColor = 'white';
            senderMessage.style.borderColor = '#ddd';
            senderPrivateKey.style.backgroundColor = 'rgba(231, 76, 60, 0.1)';
            senderPrivateKey.style.borderColor = '#e74c3c';
            
            // 移除签名提示
            const sigTip = digitalSignature.querySelector('div');
            if (sigTip) sigTip.remove();
        }, 2500);
        
        // 6. 传输消息 (3000ms)
        setTimeout(() => {
            // 创建消息流动效果
            const msgPacket = document.createElement('div');
            msgPacket.style.cssText = `
                position: absolute;
                width: 12px;
                height: 12px;
                background-color: #3498db;
                border-radius: 50%;
                box-shadow: 0 0 15px #3498db;
                left: 0;
                top: 30%;
                transform: translateY(-50%);
                opacity: 0;
                z-index: 100;
            `;
            flowContainer.appendChild(msgPacket);
            
            // 动画
            setTimeout(() => {
                msgPacket.style.transition = 'left 1s, opacity 0.5s';
                msgPacket.style.left = '100%';
                msgPacket.style.opacity = '1';
                
                setTimeout(() => {
                    msgPacket.remove();
                }, 1000);
            }, 50);
        }, 3000);
        
        // 7. 传输签名 (3300ms)
        setTimeout(() => {
            // 创建签名流动效果
            const sigPacket = document.createElement('div');
            sigPacket.style.cssText = `
                position: absolute;
                width: 12px;
                height: 12px;
                background-color: #9b59b6;
                border-radius: 50%;
                box-shadow: 0 0 15px #9b59b6;
                left: 0;
                top: 70%;
                transform: translateY(-50%);
                opacity: 0;
                z-index: 100;
            `;
            flowContainer.appendChild(sigPacket);
            
            // 动画
            setTimeout(() => {
                sigPacket.style.transition = 'left 1s, opacity 0.5s';
                sigPacket.style.left = '100%';
                sigPacket.style.opacity = '1';
                
                setTimeout(() => {
                    sigPacket.remove();
                }, 1000);
            }, 50);
        }, 3300);
        
        // 8. 接收消息和签名 (4000ms)
        setTimeout(() => {
            receiverMessage.style.display = 'block';
            receiverSignature.style.display = 'block';
            receiverMessage.style.opacity = '1';
            receiverSignature.style.opacity = '1';
            receiverMessage.style.transition = 'opacity 0.5s';
            receiverSignature.style.transition = 'opacity 0.5s';
            
            // 高亮接收的内容
            receiverMessage.style.backgroundColor = 'rgba(52, 152, 219, 0.1)';
            receiverMessage.style.borderColor = '#3498db';
            receiverSignature.style.backgroundColor = 'rgba(155, 89, 182, 0.1)';
            receiverSignature.style.borderColor = '#9b59b6';
        }, 4000);
        
        // 9. 准备使用公钥验证 (4500ms)
        setTimeout(() => {
            receiverPublicKey.style.backgroundColor = 'rgba(52, 152, 219, 0.2)';
            receiverPublicKey.style.borderColor = '#3498db';
        }, 4500);
        
        // 10. 验证签名 (5000ms)
        setTimeout(() => {
            verificationResult.style.display = 'block';
            verificationResult.style.opacity = '1';
            verificationResult.style.transition = 'opacity 0.5s';
            
            // 添加验证提示
            const verifyTip = document.createElement('div');
            verifyTip.textContent = '验证中...';
            verifyTip.style.cssText = `
                position: absolute;
                top: 50%;
                left: 10px;
                font-size: 8px;
                color: #3498db;
                animation: blink 1s infinite;
            `;
            verificationResult.appendChild(verifyTip);
        }, 5000);
        
        // 11. 验证成功 (6000ms)
        setTimeout(() => {
            // 移除验证提示
            const verifyTip = verificationResult.querySelector('div');
            if (verifyTip) verifyTip.remove();
            
            verificationResult.textContent = '验证成功！';
            verificationResult.style.backgroundColor = 'rgba(46, 204, 113, 0.1)';
            verificationResult.style.borderColor = '#27ae60';
            verificationResult.style.color = '#27ae60';
            
            // 添加详细信息
            const detail = document.createElement('div');
            detail.textContent = '消息完整且来自合法发送方';
            detail.style.cssText = `
                font-size: 8px;
                margin-top: 5px;
                color: #27ae60;
            `;
            verificationResult.appendChild(detail);
            
            // 重置高亮
            receiverMessage.style.backgroundColor = 'white';
            receiverMessage.style.borderColor = '#ddd';
            receiverSignature.style.backgroundColor = 'rgba(155, 89, 182, 0.1)';
            receiverSignature.style.borderColor = '#9b59b6';
            receiverPublicKey.style.backgroundColor = 'rgba(52, 152, 219, 0.1)';
            receiverPublicKey.style.borderColor = '#3498db';
            
            // 启用开始按钮
            if (startBtn) {
                startBtn.disabled = false;
                startBtn.style.opacity = '1';
                startBtn.style.cursor = 'pointer';
            }
        }, 6000);
        
        // 添加CSS动画
        const style = document.createElement('style');
        style.textContent = `
            @keyframes blink {
                0%, 100% { opacity: 1; }
                50% { opacity: 0.5; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // 创建数字签名数据流动效果
    function createSignatureFlowEffect(container) {
        container.innerHTML = '';
        
        for (let i = 0; i < 2; i++) {
            setTimeout(() => {
                const dataPacket = document.createElement('div');
                dataPacket.style.cssText = `
                    position: absolute;
                    width: 10px;
                    height: 10px;
                    background-color: #9b59b6;
                    border-radius: 50%;
                    box-shadow: 0 0 10px #9b59b6;
                    left: 0;
                    top: 50%;
                    transform: translateY(-50%);
                    opacity: 0;
                    z-index: 100;
                `;
                
                container.appendChild(dataPacket);
                
                // 动画
                setTimeout(() => {
                    dataPacket.style.transition = 'left 1s, opacity 0.5s';
                    dataPacket.style.left = '100%';
                    dataPacket.style.opacity = '1';
                    
                    setTimeout(() => {
                        dataPacket.remove();
                    }, 1000);
                }, 50);
            }, i * 300);
        }
    }
    
    // 显示安全WEB原理
    function showSecureWeb() {
        const modal = document.getElementById('modal');
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <div class="modal-header">
                <h3>安全WEB原理 - HTTPS加密通信</h3>
            </div>
            <div class="property-form">
                <!-- 动画展示区域 -->
                <div class="form-group">
                    <label>HTTPS网页访问过程 - 加密通信与身份验证</label>
                    <div id="secure-web-animation" style="
                        background-color: #f8f9fa;
                        padding: 15px;
                        border-radius: 8px;
                        margin-top: 10px;
                        min-height: 400px;
                        max-height: 500px;
                        overflow: hidden;
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        width: 100%;
                        max-width: 550px;
                        margin-left: auto;
                        margin-right: auto;
                    ">
                        <!-- 客户端和服务器 -->
                        <div style="
                            display: flex;
                            gap: 30px;
                            align-items: flex-start;
                            position: relative;
                            width: 100%;
                            justify-content: center;
                        ">
                            <!-- 客户端 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">客户端</h5>
                                <div id="client-https" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #3498db;
                                    border-radius: 8px;
                                    padding: 10px;
                                    min-height: 180px;
                                    width: 180px;
                                    display: flex;
                                    flex-direction: column;
                                    gap: 8px;
                                ">
                                    <!-- 浏览器图标 -->
                                    <div style="text-align: center; font-size: 30px; margin-bottom: 5px;">
                                        🌐
                                    </div>
                                    <!-- 客户端证书 -->
                                    <div id="client-cert" style="
                                        background-color: rgba(52, 152, 219, 0.1);
                                        border: 1px solid #3498db;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                        display: none;
                                    ">
                                        <strong>客户端公钥</strong><br/>
                                        [客户端证书]
                                    </div>
                                    <!-- 加密数据 -->
                                    <div id="client-encrypted-data" style="
                                        background-color: rgba(155, 89, 182, 0.1);
                                        border: 1px solid #9b59b6;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                        display: none;
                                    ">
                                        <strong>加密数据</strong><br/>
                                        [加密请求]
                                    </div>
                                </div>
                            </div>
                            
                            <!-- 传输路径 -->
                            <div style="
                                position: relative;
                                width: 100px;
                                height: 100px;
                                display: flex;
                                align-items: center;
                                justify-content: center;
                            ">
                                <div style="
                                    width: 100%;
                                    height: 4px;
                                    background-color: #bdc3c7;
                                    position: absolute;
                                    top: 50%;
                                    left: 0;
                                "></div>
                                <!-- 数据流动效果容器 -->
                                <div id="https-flow-container" style="
                                    position: absolute;
                                    top: 0;
                                    left: 0;
                                    width: 100%;
                                    height: 100%;
                                    pointer-events: none;
                                    z-index: 10;
                                "></div>
                            </div>
                            
                            <!-- 服务器 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">服务器</h5>
                                <div id="server-https" style="
                                    background-color: #ecf0f1;
                                    border: 2px solid #27ae60;
                                    border-radius: 8px;
                                    padding: 10px;
                                    min-height: 180px;
                                    width: 180px;
                                    display: flex;
                                    flex-direction: column;
                                    gap: 8px;
                                ">
                                    <!-- 服务器图标 -->
                                    <div style="text-align: center; font-size: 30px; margin-bottom: 5px;">
                                        🖥️
                                    </div>
                                    <!-- 服务器证书 -->
                                    <div id="server-cert" style="
                                        background-color: rgba(46, 204, 113, 0.1);
                                        border: 1px solid #27ae60;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                    ">
                                        <strong>服务器证书</strong><br/>
                                        [SSL/TLS证书]
                                    </div>
                                    <!-- 加密响应 -->
                                    <div id="server-encrypted-data" style="
                                        background-color: rgba(155, 89, 182, 0.1);
                                        border: 1px solid #9b59b6;
                                        border-radius: 4px;
                                        padding: 8px;
                                        font-size: 9px;
                                        display: none;
                                    ">
                                        <strong>加密响应</strong><br/>
                                        [加密网页内容]
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 通信步骤说明 -->
                        <div id="https-steps" style="
                            margin-top: 15px;
                            background-color: white;
                            border: 1px solid #ddd;
                            border-radius: 8px;
                            padding: 10px;
                            width: 100%;
                            max-width: 500px;
                        ">
                            <h6 style="margin-bottom: 8px; font-size: 12px;">通信步骤:</h6>
                            <div id="step-1" style="
                                margin-bottom: 5px;
                                padding: 6px;
                                border-left: 3px solid #95a5a6;
                                font-size: 10px;
                            ">
                                1. 客户端发起HTTPS连接请求
                            </div>
                            <div id="step-2" style="
                                margin-bottom: 5px;
                                padding: 6px;
                                border-left: 3px solid #95a5a6;
                                font-size: 10px;
                            ">
                                2. 服务器发送SSL/TLS证书
                            </div>
                            <div id="step-3" style="
                                margin-bottom: 5px;
                                padding: 6px;
                                border-left: 3px solid #95a5a6;
                                font-size: 10px;
                            ">
                                3. 客户端验证证书有效性
                            </div>
                            <div id="step-4" style="
                                margin-bottom: 5px;
                                padding: 6px;
                                border-left: 3px solid #95a5a6;
                                font-size: 10px;
                            ">
                                4. 客户端生成会话密钥并加密发送
                            </div>
                            <div id="step-5" style="
                                margin-bottom: 5px;
                                padding: 6px;
                                border-left: 3px solid #95a5a6;
                                font-size: 10px;
                            ">
                                5. 服务器解密会话密钥
                            </div>
                            <div id="step-6" style="
                                margin-bottom: 5px;
                                padding: 6px;
                                border-left: 3px solid #95a5a6;
                                font-size: 10px;
                            ">
                                6. 双方使用会话密钥加密通信
                            </div>
                        </div>
                    </div>
                    
                    <!-- 控制按钮 -->
                    <div class="form-actions" style="margin-top: 15px; display: flex; gap: 10px; justify-content: center;">
                        <button id="start-https-btn" class="btn-confirm">开始</button>
                        <button id="close-https-btn" class="btn-cancel">关闭</button>
                    </div>
                </div>
            </div>
        `;
        
        modal.style.display = 'block';
        initModalDrag();
        
        // 添加事件监听器
        const startBtn = document.getElementById('start-https-btn');
        const closeBtn = document.getElementById('close-https-btn');
        
        if (startBtn) {
            startBtn.addEventListener('click', startSecureWebAnimation);
        }
        
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                document.getElementById('modal').style.display = 'none';
            });
        }
    }
    
    // HTTPS动画函数
    window.startSecureWebAnimation = function() {
        console.log('开始安全WEB动画');
        
        // 禁用开始按钮
        const startBtn = document.getElementById('start-https-btn');
        if (startBtn) {
            startBtn.disabled = true;
            startBtn.style.opacity = '0.5';
            startBtn.style.cursor = 'not-allowed';
        }
        
        const clientHttps = document.getElementById('client-https');
        const serverHttps = document.getElementById('server-https');
        const clientCert = document.getElementById('client-cert');
        const clientEncryptedData = document.getElementById('client-encrypted-data');
        const serverCert = document.getElementById('server-cert');
        const serverEncryptedData = document.getElementById('server-encrypted-data');
        const flowContainer = document.getElementById('https-flow-container');
        
        // 1. 客户端准备发起请求 (500ms)
        setTimeout(() => {
            highlightStep(1);
            clientHttps.style.borderColor = '#3498db';
            clientHttps.style.boxShadow = '0 0 10px rgba(52, 152, 219, 0.5)';
        }, 500);
        
        // 2. 客户端发起HTTPS请求 (1000ms)
        setTimeout(() => {
            // 创建请求数据包
            const requestPacket = document.createElement('div');
            requestPacket.style.cssText = `
                position: absolute;
                width: 12px;
                height: 12px;
                background-color: #3498db;
                border-radius: 50%;
                box-shadow: 0 0 15px #3498db;
                left: 0;
                top: 30%;
                transform: translateY(-50%);
                opacity: 0;
                z-index: 100;
            `;
            flowContainer.appendChild(requestPacket);
            
            // 动画
            setTimeout(() => {
                requestPacket.style.transition = 'left 1s, opacity 0.5s';
                requestPacket.style.left = '100%';
                requestPacket.style.opacity = '1';
                
                setTimeout(() => {
                    requestPacket.remove();
                }, 1000);
            }, 50);
            
            // 添加请求提示
            const requestTip = document.createElement('div');
            requestTip.textContent = 'HTTPS请求';
            requestTip.style.cssText = `
                position: absolute;
                bottom: 10px;
                left: 50%;
                transform: translateX(-50%);
                font-size: 8px;
                color: #3498db;
                animation: fadeInOut 2s ease;
            `;
            clientHttps.appendChild(requestTip);
            
            setTimeout(() => requestTip.remove(), 2000);
        }, 1000);
        
        // 3. 服务器准备响应 (1500ms)
        setTimeout(() => {
            highlightStep(2);
            serverHttps.style.borderColor = '#27ae60';
            serverHttps.style.boxShadow = '0 0 10px rgba(46, 204, 113, 0.5)';
        }, 1500);
        
        // 4. 服务器发送SSL/TLS证书 (2000ms)
        setTimeout(() => {
            // 创建证书数据包
            const certPacket = document.createElement('div');
            certPacket.style.cssText = `
                position: absolute;
                width: 12px;
                height: 12px;
                background-color: #27ae60;
                border-radius: 50%;
                box-shadow: 0 0 15px #27ae60;
                left: 100%;
                top: 50%;
                transform: translateY(-50%);
                opacity: 0;
                z-index: 100;
            `;
            flowContainer.appendChild(certPacket);
            
            // 动画
            setTimeout(() => {
                certPacket.style.transition = 'left 1s, opacity 0.5s';
                certPacket.style.left = '0';
                certPacket.style.opacity = '1';
                
                setTimeout(() => {
                    certPacket.remove();
                }, 1000);
            }, 50);
            
            // 高亮服务器证书
            serverCert.style.backgroundColor = 'rgba(46, 204, 113, 0.2)';
            serverCert.style.borderColor = '#27ae60';
        }, 2000);
        
        // 5. 客户端验证证书 (2500ms)
        setTimeout(() => {
            highlightStep(3);
            
            // 显示客户端证书
            clientCert.style.display = 'block';
            clientCert.style.opacity = '1';
            clientCert.style.transition = 'opacity 0.5s';
            
            // 添加验证提示
            const verifyTip = document.createElement('div');
            verifyTip.textContent = '验证证书...';
            verifyTip.style.cssText = `
                position: absolute;
                bottom: 10px;
                left: 50%;
                transform: translateX(-50%);
                font-size: 8px;
                color: #3498db;
                animation: blink 1s infinite;
            `;
            clientHttps.appendChild(verifyTip);
            
            setTimeout(() => verifyTip.remove(), 1000);
        }, 2500);
        
        // 6. 客户端生成会话密钥 (3000ms)
        setTimeout(() => {
            // 添加密钥生成提示
            const keyTip = document.createElement('div');
            keyTip.textContent = '生成会话密钥...';
            keyTip.style.cssText = `
                position: absolute;
                bottom: 10px;
                left: 50%;
                transform: translateX(-50%);
                font-size: 8px;
                color: #9b59b6;
                animation: blink 1s infinite;
            `;
            clientHttps.appendChild(keyTip);
            
            setTimeout(() => keyTip.remove(), 1000);
        }, 3000);
        
        // 7. 客户端发送加密会话密钥 (3500ms)
        setTimeout(() => {
            highlightStep(4);
            
            // 显示加密数据
            clientEncryptedData.style.display = 'block';
            clientEncryptedData.style.opacity = '1';
            clientEncryptedData.style.transition = 'opacity 0.5s';
            
            // 创建加密密钥数据包
            const keyPacket = document.createElement('div');
            keyPacket.style.cssText = `
                position: absolute;
                width: 12px;
                height: 12px;
                background-color: #9b59b6;
                border-radius: 50%;
                box-shadow: 0 0 15px #9b59b6;
                left: 0;
                top: 70%;
                transform: translateY(-50%);
                opacity: 0;
                z-index: 100;
            `;
            flowContainer.appendChild(keyPacket);
            
            // 动画
            setTimeout(() => {
                keyPacket.style.transition = 'left 1s, opacity 0.5s';
                keyPacket.style.left = '100%';
                keyPacket.style.opacity = '1';
                
                setTimeout(() => {
                    keyPacket.remove();
                }, 1000);
            }, 50);
            
            // 添加密钥发送提示
            const sendKeyTip = document.createElement('div');
            sendKeyTip.textContent = '发送加密密钥';
            sendKeyTip.style.cssText = `
                position: absolute;
                bottom: 10px;
                left: 50%;
                transform: translateX(-50%);
                font-size: 8px;
                color: #9b59b6;
                animation: fadeInOut 2s ease;
            `;
            clientHttps.appendChild(sendKeyTip);
            
            setTimeout(() => sendKeyTip.remove(), 2000);
        }, 3500);
        
        // 8. 服务器解密会话密钥 (4000ms)
        setTimeout(() => {
            highlightStep(5);
            
            // 添加解密提示
            const decryptTip = document.createElement('div');
            decryptTip.textContent = '解密密钥...';
            decryptTip.style.cssText = `
                position: absolute;
                bottom: 10px;
                left: 50%;
                transform: translateX(-50%);
                font-size: 8px;
                color: #27ae60;
                animation: blink 1s infinite;
            `;
            serverHttps.appendChild(decryptTip);
            
            setTimeout(() => decryptTip.remove(), 1000);
        }, 4000);
        
        // 9. 服务器准备加密响应 (4500ms)
        setTimeout(() => {
            // 添加响应准备提示
            const prepareTip = document.createElement('div');
            prepareTip.textContent = '准备加密响应...';
            prepareTip.style.cssText = `
                position: absolute;
                bottom: 10px;
                left: 50%;
                transform: translateX(-50%);
                font-size: 8px;
                color: #27ae60;
                animation: blink 1s infinite;
            `;
            serverHttps.appendChild(prepareTip);
            
            setTimeout(() => prepareTip.remove(), 1000);
        }, 4500);
        
        // 10. 服务器发送加密响应 (5000ms)
        setTimeout(() => {
            highlightStep(6);
            
            // 显示服务器加密数据
            serverEncryptedData.style.display = 'block';
            serverEncryptedData.style.opacity = '1';
            serverEncryptedData.style.transition = 'opacity 0.5s';
            
            // 创建加密响应数据包
            const responsePacket = document.createElement('div');
            responsePacket.style.cssText = `
                position: absolute;
                width: 12px;
                height: 12px;
                background-color: #9b59b6;
                border-radius: 50%;
                box-shadow: 0 0 15px #9b59b6;
                left: 100%;
                top: 70%;
                transform: translateY(-50%);
                opacity: 0;
                z-index: 100;
            `;
            flowContainer.appendChild(responsePacket);
            
            // 动画
            setTimeout(() => {
                responsePacket.style.transition = 'left 1s, opacity 0.5s';
                responsePacket.style.left = '0';
                responsePacket.style.opacity = '1';
                
                setTimeout(() => {
                    responsePacket.remove();
                }, 1000);
            }, 50);
            
            // 添加响应提示
            const responseTip = document.createElement('div');
            responseTip.textContent = '加密响应';
            responseTip.style.cssText = `
                position: absolute;
                bottom: 10px;
                left: 50%;
                transform: translateX(-50%);
                font-size: 8px;
                color: #9b59b6;
                animation: fadeInOut 2s ease;
            `;
            serverHttps.appendChild(responseTip);
            
            setTimeout(() => responseTip.remove(), 2000);
        }, 5000);
        
        // 11. 完成通信 (6000ms)
        setTimeout(() => {
            // 重置边框颜色
            clientHttps.style.borderColor = '#3498db';
            clientHttps.style.boxShadow = 'none';
            serverHttps.style.borderColor = '#27ae60';
            serverHttps.style.boxShadow = 'none';
            
            // 启用开始按钮
            if (startBtn) {
                startBtn.disabled = false;
                startBtn.style.opacity = '1';
                startBtn.style.cursor = 'pointer';
            }
        }, 6000);
        
        // 添加CSS动画
        const style = document.createElement('style');
        style.textContent = `
            @keyframes blink {
                0%, 100% { opacity: 1; }
                50% { opacity: 0.5; }
            }
            @keyframes fadeInOut {
                0%, 100% { opacity: 0; }
                50% { opacity: 1; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // 高亮显示步骤
    function highlightStep(stepNumber) {
        for (let i = 1; i <= 6; i++) {
            const step = document.getElementById(`step-${i}`);
            if (i < stepNumber) {
                // 已完成的步骤
                step.style.borderLeftColor = '#27ae60';
                step.style.backgroundColor = 'rgba(46, 204, 113, 0.1)';
            } else if (i === stepNumber) {
                // 当前步骤
                step.style.borderLeftColor = '#3498db';
                step.style.backgroundColor = 'rgba(52, 152, 219, 0.1)';
                step.style.fontWeight = 'bold';
            } else {
                // 未完成的步骤
                step.style.borderLeftColor = '#95a5a6';
                step.style.backgroundColor = 'transparent';
            }
        }
    }
    
    // 创建HTTPS数据流动效果
    function createHttpsFlowEffect(container, color, direction) {
        container.innerHTML = '';
        
        const dataPacket = document.createElement('div');
        dataPacket.style.cssText = `
            position: absolute;
            width: 10px;
            height: 10px;
            background-color: ${color};
            border-radius: 50%;
            box-shadow: 0 0 10px ${color};
            ${direction === 'left' ? 'left: 100%;' : 'left: 0;'} 
            top: 50%;
            transform: translateY(-50%);
            opacity: 0;
            z-index: 100;
        `;
        
        container.appendChild(dataPacket);
        
        // 动画
        setTimeout(() => {
            dataPacket.style.transition = 'left 1s, opacity 0.5s';
            dataPacket.style.left = direction === 'left' ? '0' : '100%';
            dataPacket.style.opacity = '1';
            
            setTimeout(() => {
                dataPacket.remove();
            }, 1000);
        }, 50);
    }
    
    // 显示分层模型原理
    function showLayeredModel() {
        const modal = document.getElementById('modal');
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <div class="modal-header">
                <h3>网络分层模型 - 数据包传输过程</h3>
            </div>
            <div class="property-form">
                <!-- 动画展示区域 -->
                <div class="form-group">
                    <label>ISO七层模型 - 端到端数据传输</label>
                    <div id="layered-model-animation" style="
                        background-color: #f8f9fa;
                        padding: 15px;
                        border-radius: 8px;
                        margin-top: 10px;
                        min-height: 400px;
                        max-height: 500px;
                        overflow-y: auto;
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        width: 100%;
                    ">
                        <!-- 发送端和接收端模型 -->
                        <div style="
                            display: flex;
                            gap: 60px;
                            align-items: flex-start;
                            position: relative;
                            width: 100%;
                            justify-content: center;
                        ">
                            <!-- 发送端ISO模型 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">发送端</h5>
                                <div id="sender-iso-model" style="
                                    display: flex;
                                    flex-direction: column;
                                    gap: 3px;
                                    position: relative;
                                ">
                                    ${createLayer('应用层', 'Application', '#c0392b')}
                                    ${createLayer('表示层', 'Presentation', '#e67e22')}
                                    ${createLayer('会话层', 'Session', '#f39c12')}
                                    ${createLayer('传输层', 'Transport', '#27ae60')}
                                    ${createLayer('网络层', 'Network', '#2980b9')}
                                    ${createLayer('数据链路层', 'Data Link', '#8e44ad')}
                                    ${createLayer('物理层', 'Physical', '#16a085')}
                                    <!-- 发送端数据包 -->
                                    <div id="sender-data-packet" style="
                                        position: absolute;
                                        left: 50%;
                                        top: 0;
                                        transform: translateX(-50%);
                                        width: 85%;
                                        height: 20px;
                                        background-color: rgba(255, 255, 255, 0.95);
                                        border: 2px solid #2980b9;
                                        border-radius: 4px;
                                        display: flex;
                                        align-items: center;
                                        justify-content: center;
                                        font-size: 11px;
                                        font-weight: bold;
                                        color: #2980b9;
                                        z-index: 10;
                                        display: none;
                                    ">数据</div>
                                </div>
                            </div>
                            
                            <!-- 接收端ISO模型 -->
                            <div style="display: flex; flex-direction: column; align-items: center;">
                                <h5 style="margin-bottom: 10px; font-size: 14px;">接收端</h5>
                                <div id="receiver-iso-model" style="
                                    display: flex;
                                    flex-direction: column;
                                    gap: 3px;
                                    position: relative;
                                ">
                                    ${createLayer('应用层', 'Application', '#c0392b')}
                                    ${createLayer('表示层', 'Presentation', '#e67e22')}
                                    ${createLayer('会话层', 'Session', '#f39c12')}
                                    ${createLayer('传输层', 'Transport', '#27ae60')}
                                    ${createLayer('网络层', 'Network', '#2980b9')}
                                    ${createLayer('数据链路层', 'Data Link', '#8e44ad')}
                                    ${createLayer('物理层', 'Physical', '#16a085')}
                                    <!-- 接收端数据包 -->
                                    <div id="receiver-data-packet" style="
                                        position: absolute;
                                        left: 50%;
                                        top: 0;
                                        transform: translateX(-50%);
                                        width: 85%;
                                        height: 20px;
                                        background-color: rgba(255, 255, 255, 0.95);
                                        border: 2px solid #2980b9;
                                        border-radius: 4px;
                                        display: flex;
                                        align-items: center;
                                        justify-content: center;
                                        font-size: 11px;
                                        font-weight: bold;
                                        color: #2980b9;
                                        z-index: 10;
                                        display: none;
                                    ">数据</div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 网络连接 -->
                        <div style="
                            width: 100%;
                            height: 60px;
                            margin-top: 20px;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            position: relative;
                        ">
                            <div style="
                                width: 80%;
                                height: 4px;
                                background-color: #bdc3c7;
                                position: relative;
                            ">
                                <!-- 网络节点 -->
                                <div style="
                                    position: absolute;
                                    left: 20%;
                                    top: -10px;
                                    width: 24px;
                                    height: 24px;
                                    background-color: #2980b9;
                                    border-radius: 50%;
                                    display: flex;
                                    align-items: center;
                                    justify-content: center;
                                    color: white;
                                    font-size: 10px;
                                    font-weight: bold;
                                ">R1</div>
                                <div style="
                                    position: absolute;
                                    left: 50%;
                                    top: -10px;
                                    width: 24px;
                                    height: 24px;
                                    background-color: #2980b9;
                                    border-radius: 50%;
                                    display: flex;
                                    align-items: center;
                                    justify-content: center;
                                    color: white;
                                    font-size: 10px;
                                    font-weight: bold;
                                ">R2</div>
                                <div style="
                                    position: absolute;
                                    left: 80%;
                                    top: -10px;
                                    width: 24px;
                                    height: 24px;
                                    background-color: #2980b9;
                                    border-radius: 50%;
                                    display: flex;
                                    align-items: center;
                                    justify-content: center;
                                    color: white;
                                    font-size: 10px;
                                    font-weight: bold;
                                ">R3</div>
                                <!-- 数据包传输动画 -->
                                <div id="network-packet" style="
                                    position: absolute;
                                    left: 0;
                                    top: -8px;
                                    width: 20px;
                                    height: 20px;
                                    background-color: rgba(41, 128, 185, 0.9);
                                    border: 2px solid #1f618d;
                                    border-radius: 50%;
                                    display: none;
                                    z-index: 5;
                                "></div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- 控制按钮 -->
                <div class="form-actions" style="margin-top: 15px; display: flex; gap: 10px; justify-content: center;">
                    <button id="start-layered-btn" class="btn-confirm">开始</button>
                    <button id="close-layered-btn" class="btn-cancel">关闭</button>
                </div>
            </div>
        `;
        
        modal.style.display = 'block';
        initModalDrag();
        
        // 添加事件监听器
        const startBtn = document.getElementById('start-layered-btn');
        const closeBtn = document.getElementById('close-layered-btn');
        
        if (startBtn) {
            startBtn.addEventListener('click', startLayeredModelAnimation);
        }
        
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                document.getElementById('modal').style.display = 'none';
            });
        }
    }
    
    // 创建模型层次（高度1/2，宽度优化）
    function createLayer(layerName, englishName, color) {
        return `
            <div style="
                background-color: ${color};
                color: white;
                padding: 6px 15px;
                border-radius: 4px;
                text-align: center;
                box-shadow: 0 2px 5px rgba(0,0,0,0.15);
                transition: all 0.3s;
                position: relative;
                overflow: hidden;
                width: 250px;
                height: 35px;
                display: flex;
                flex-direction: column;
                justify-content: center;
                border: 1px solid rgba(255,255,255,0.2);
            ">
                <div style="font-weight: bold; margin-bottom: 1px; font-size: 13px;">${layerName}</div>
                <div style="font-size: 9px; opacity: 0.95;">${englishName}</div>
            </div>
        `;
    }
    
    // 更新模型显示 - 不再需要，因为现在只显示ISO七层模型
    function updateModelView() {
        // 现在只显示ISO七层模型，所以这个函数不再需要
    }
    
    // 开始分层模型动画（完整的端到端传输）
    function startLayeredModelAnimation() {
        console.log('开始端到端数据传输动画');
        isAnimating = true;
        
        // 禁用开始按钮
        const startBtn = document.getElementById('start-layered-btn');
        if (startBtn) {
            startBtn.disabled = true;
            startBtn.style.opacity = '0.5';
            startBtn.style.cursor = 'not-allowed';
        }
        
        // 1. 发送端封装动画
        animateSenderEncapsulation();
        
        // 2. 网络传输动画
        principleTimers.push(setTimeout(() => {
            animateNetworkTransmission();
        }, 7 * 1200 + 1000));
        
        // 3. 接收端解包动画
        principleTimers.push(setTimeout(() => {
            animateReceiverDecapsulation();
        }, 7 * 1200 + 1000 + 2000));
        
        // 4. 动画完成后重新启用开始按钮
        principleTimers.push(setTimeout(() => {
            if (startBtn) {
                startBtn.disabled = false;
                startBtn.style.opacity = '1';
                startBtn.style.cursor = 'pointer';
            }
            isAnimating = false;
        }, 7 * 1200 + 1000 + 2000 + 7 * 1200)); // 动画全部完成后
    }
    
    // 发送端封装动画
    function animateSenderEncapsulation() {
        // 只选择ISO模型的7个层，排除数据包元素
        const layers = Array.from(document.querySelectorAll('#sender-iso-model > div'))
            .filter(div => !div.id || div.id !== 'sender-data-packet');
        const packet = document.getElementById('sender-data-packet');
        
        if (!packet) return;
        
        // 重置所有层的状态
        layers.forEach(layer => {
            layer.style.transform = 'scale(1)';
            layer.style.boxShadow = '0 2px 5px rgba(0,0,0,0.15)';
        });
        
        // 显示数据包
        packet.style.display = 'flex';
        packet.style.top = '0px';
        packet.style.opacity = '1';
        packet.innerHTML = '数据';
        
        // 动画序列：从应用层开始向下封装
        layers.forEach((layer, index) => {
            setTimeout(() => {
                // 层高亮
                layer.style.transform = 'scale(1.1)';
                layer.style.boxShadow = '0 3px 12px rgba(0,0,0,0.25)';
                
                // 移动数据包到当前层
                const layerTop = layer.offsetTop;
                const layerHeight = layer.offsetHeight;
                packet.style.top = (layerTop + (layerHeight - 20) / 2) + 'px';
                
                // 更新数据包内容 - 正确的封装过程
                // 从应用层开始，逐层添加头部，最终变为比特流
                let content = '数据';
                
                // 根据当前处理的层更新数据包内容
                if (index === 0) { // 应用层
                    content = '数据';
                } else if (index === 1) { // 表示层
                    content = '表示头+数据';
                } else if (index === 2) { // 会话层
                    content = '会话头+表示头+数据';
                } else if (index === 3) { // 传输层
                    content = 'TCP头+会话头+表示头+数据';
                } else if (index === 4) { // 网络层
                    content = 'IP头+TCP头+会话头+表示头+数据';
                } else if (index === 5) { // 数据链路层
                    content = '帧头+IP头+TCP头+会话头+表示头+数据';
                } else if (index === 6) { // 物理层
                    content = '比特流';
                }
                
                packet.innerHTML = content;
                
                setTimeout(() => {
                    layer.style.transform = 'scale(1)';
                    layer.style.boxShadow = '0 2px 5px rgba(0,0,0,0.15)';
                }, 1000);
            }, index * 1200);
        });
        
        // 动画结束后隐藏发送端数据包
        setTimeout(() => {
            packet.style.opacity = '0';
            setTimeout(() => {
                packet.style.display = 'none';
            }, 500);
        }, layers.length * 1200);
    }
    
    // 网络传输动画
    function animateNetworkTransmission() {
        const networkPacket = document.getElementById('network-packet');
        
        // 检查网络数据包元素是否存在
        if (!networkPacket) {
            console.error('network-packet element not found');
            return;
        }
        
        // 显示网络数据包，设置内容为比特流
        networkPacket.style.display = 'block';
        networkPacket.style.opacity = '1';
        networkPacket.style.left = '0%';
        networkPacket.innerHTML = '';
        
        // 动画：数据包从发送端到接收端
        let startTime = null;
        const duration = 2000; // 2秒
        
        function animate(timestamp) {
            if (!startTime) startTime = timestamp;
            const progress = timestamp - startTime;
            const percentage = Math.min(progress / duration, 1);
            
            networkPacket.style.left = `${percentage * 100}%`;
            
            if (progress < duration) {
                requestAnimationFrame(animate);
            } else {
                // 传输完成，隐藏数据包
                networkPacket.style.opacity = '0';
                setTimeout(() => {
                    networkPacket.style.display = 'none';
                }, 500);
            }
        }
        
        requestAnimationFrame(animate);
    }
    
    // 接收端解包动画
    function animateReceiverDecapsulation() {
        // 只选择ISO模型的7个层，排除数据包元素
        const allLayers = Array.from(document.querySelectorAll('#receiver-iso-model > div'))
            .filter(div => !div.id || div.id !== 'receiver-data-packet');
        // 从物理层开始处理，所以需要反转层数组
        const layers = allLayers.reverse();
        const packet = document.getElementById('receiver-data-packet');
        
        if (!packet) return;
        
        // 重置所有层的状态
        Array.from(allLayers).forEach(layer => {
            layer.style.transform = 'scale(1)';
            layer.style.boxShadow = '0 2px 5px rgba(0,0,0,0.15)';
        });
        
        // 显示数据包在最底层（物理层）
        packet.style.display = 'flex';
        packet.style.opacity = '1';
        packet.innerHTML = '比特流';
        
        // 设置数据包初始位置在物理层
        const physicalLayer = layers[0]; // 反转后第一层是物理层
        const layerTop = physicalLayer.offsetTop;
        const layerHeight = physicalLayer.offsetHeight;
        packet.style.top = (layerTop + (layerHeight - 20) / 2) + 'px';
        
        // 定义解包过程的内容变化（从物理层到应用层）
        const decapsulationContent = [
            '比特流',                 // 物理层：接收比特流
            '帧头+IP头+TCP头+会话头+表示头+数据', // 数据链路层：从比特流中提取帧
            'IP头+TCP头+会话头+表示头+数据', // 网络层：移除帧头
            'TCP头+会话头+表示头+数据', // 传输层：移除IP头
            '会话头+表示头+数据',     // 会话层：移除TCP头
            '表示头+数据',           // 表示层：移除会话头
            '数据'                  // 应用层：移除表示头，得到最终数据
        ];
        
        // 动画序列：从物理层开始向上解包（已反转数组）
        layers.forEach((layer, i) => {
            setTimeout(() => {
                // 层高亮
                layer.style.transform = 'scale(1.1)';
                layer.style.boxShadow = '0 3px 12px rgba(0,0,0,0.25)';
                
                // 移动数据包到当前层
                const layerTop = layer.offsetTop;
                const layerHeight = layer.offsetHeight;
                packet.style.top = (layerTop + (layerHeight - 20) / 2) + 'px';
                
                // 更新数据包内容 - 解包过程是封装的逆过程
                packet.innerHTML = decapsulationContent[i] || '比特流';
                
                setTimeout(() => {
                    layer.style.transform = 'scale(1)';
                    layer.style.boxShadow = '0 2px 5px rgba(0,0,0,0.15)';
                }, 1000);
            }, i * 1200);
        });
        
        // 动画结束后保持数据包显示
        setTimeout(() => {
            packet.style.opacity = '0.95';
        }, layers.length * 1200);
    }
    
    // 键盘事件 - Delete键删除选中设备或连接
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Delete' || e.key === 'Del' || e.keyCode === 46) {
            // 检查当前焦点是否在可编辑元素上（输入框、文本区域等）
            const activeElement = document.activeElement;
            const isEditable = activeElement && (
                activeElement.tagName === 'INPUT' ||
                activeElement.tagName === 'TEXTAREA' ||
                activeElement.isContentEditable ||
                activeElement.closest('input, textarea, [contenteditable="true"]')
            );
            
            // 如果不在可编辑元素上，且不在仿真运行时，可以删除选中的设备或连接
            if (!isEditable && !isSimulating) {
                if (selectedConnection) {
                    e.preventDefault(); // 防止默认行为
                    deleteConnection();
                } else if (selectedDevice) {
                    e.preventDefault(); // 防止默认行为
                    deleteDevice(selectedDevice.id);
                }
            }
        }
    });
}





// 创建拖动预览
function createDragPreview(deviceElement, x, y) {
    // 移除之前的预览
    cleanupDragPreview();
    
    // 创建预览元素
    dragPreview = document.createElement('div');
    dragPreview.className = 'drag-preview';
    
    // 复制设备项的内容
    const deviceIcon = deviceElement.querySelector('.device-icon');
    const deviceName = deviceElement.querySelector('span');
    
    if (deviceIcon && deviceName) {
        dragPreview.innerHTML = `
            <div class="device-icon">${deviceIcon.innerHTML}</div>
            <span>${deviceName.textContent}</span>
        `;
    }
    
    // 设置初始位置
    updateDragPreview(x, y);
    
    // 添加到文档
    document.body.appendChild(dragPreview);
}

// 更新拖动预览位置
function updateDragPreview(x, y) {
    if (dragPreview) {
        dragPreview.style.left = `${x - 30}px`;
        dragPreview.style.top = `${y - 40}px`;
    }
}

// 清理拖动预览
function cleanupDragPreview() {
    if (dragPreview && dragPreview.parentNode) {
        dragPreview.parentNode.removeChild(dragPreview);
        dragPreview = null;
    }
}

// 清理触摸拖动状态
function cleanupTouchDrag() {
    // 移除所有设备项的触摸激活状态
    document.querySelectorAll('.device-item').forEach(item => {
        item.classList.remove('touch-active');
    });
    
    // 移除拖动预览
    cleanupDragPreview();
    
    // 重置状态
    isTouchDragging = false;
    touchDeviceType = null;
}

// 添加设备
function addDevice(type, x, y) {
    // 仿真运行时不能添加设备
    if (isSimulating) {
        alert('仿真运行时不能添加设备，请先停止仿真');
        return null;
    }
    
    // 检查设备类型是否存在
    const deviceData = DEVICE_TYPES[type];
    if (!deviceData) {
        log(`添加设备失败: 未知设备类型 ${type}`);
        return null;
    }
    const deviceId = `device_${Date.now()}_${Math.floor(Math.random() * 1000)}`;
    
    // 先计算设备名称，以便在标签中使用
    // 统计同类型设备的数量，按设备类型分别计数
    const sameTypeDevices = devices.filter(d => d.type === type);
    const deviceNumber = sameTypeDevices.length + 1;
    const deviceName = `${deviceData.name}_${deviceNumber}`;
    
    // 创建设备组
    const deviceGroup = document.createElementNS("http://www.w3.org/2000/svg", "g");
    deviceGroup.setAttribute("class", "device");
    deviceGroup.setAttribute("id", deviceId);
    
    // 设备背景
    const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
    rect.setAttribute("width", deviceData.width);
    rect.setAttribute("height", deviceData.height);
    rect.setAttribute("fill", "#e3f2fd");
    rect.setAttribute("stroke", "#3498db");
    rect.setAttribute("stroke-width", "2");
    rect.setAttribute("rx", "8");
    rect.setAttribute("ry", "8");
    deviceGroup.appendChild(rect);
    
    // 设备图标
    if (deviceData.icon.includes('<img') || deviceData.icon.includes('<svg')) {
        // 创建foreignObject来支持HTML内容（包括img和svg）
        const foreignObject = document.createElementNS("http://www.w3.org/2000/svg", "foreignObject");
        foreignObject.setAttribute("x", deviceData.width / 2 - 16);
        foreignObject.setAttribute("y", deviceData.height / 2 - 16);
        foreignObject.setAttribute("width", "32");
        foreignObject.setAttribute("height", "32");
        
        // 创建div容器
        const div = document.createElement("div");
        div.innerHTML = deviceData.icon;
        foreignObject.appendChild(div);
        deviceGroup.appendChild(foreignObject);
    } else {
        // 文本图标（emoji）
        const iconText = document.createElementNS("http://www.w3.org/2000/svg", "text");
        iconText.textContent = deviceData.icon;
        iconText.setAttribute("x", deviceData.width / 2);
        iconText.setAttribute("y", deviceData.height / 2 + 5);
        iconText.setAttribute("text-anchor", "middle");
        iconText.setAttribute("font-size", "32");
        iconText.setAttribute("font-family", "Arial");
        deviceGroup.appendChild(iconText);
    }
    
    // 设备标签 - 使用计算出的设备名称
    const labelText = document.createElementNS("http://www.w3.org/2000/svg", "text");
    labelText.textContent = deviceName;
    labelText.setAttribute("class", "device-label");
    labelText.setAttribute("x", deviceData.width / 2);
    labelText.setAttribute("y", deviceData.height + 15);
    labelText.setAttribute("text-anchor", "middle");
    labelText.setAttribute("font-size", "12");
    labelText.setAttribute("font-weight", "bold");
    labelText.setAttribute("fill", "#2c3e50");
    deviceGroup.appendChild(labelText);
    
    // 设置位置
    deviceGroup.setAttribute("transform", `translate(${x - deviceData.width / 2}, ${y - deviceData.height / 2})`);
    
    // 添加到画布
    canvas.appendChild(deviceGroup);
    
    // 添加拖拽功能
    makeDraggable(deviceGroup);
    
    // 添加点击事件
    deviceGroup.addEventListener('click', function(e) {
        e.stopPropagation();
        // 使用当前元素的ID，而不是局部变量deviceId，确保ID变更后仍能正常工作
        handleDeviceClick(this.id);
    });
    
    // 添加触摸事件支持
    deviceGroup.addEventListener('touchstart', function(e) {
        e.stopPropagation();
        // 不要阻止默认行为，否则可能影响后续的点击事件
        // 使用当前元素的ID，而不是局部变量deviceId，确保ID变更后仍能正常工作
        handleDeviceClick(this.id);
    });
    
    // 创建设备对象
    const device = {
        id: deviceId,
        type: type,
        name: deviceName,
        x: x - deviceData.width / 2,
        y: y - deviceData.height / 2,
        width: deviceData.width,
        height: deviceData.height,
        element: deviceGroup,
        properties: JSON.parse(JSON.stringify(deviceData.defaultProps)), // 使用深拷贝确保每个设备都有独立的属性
        connections: []
    };
    
    // 添加到设备列表
    devices.push(device);
    
    // 确保设备标签显示正确的名称（防止被其他地方覆盖）
    const label = device.element.querySelector('.device-label');
    if (label) {
        label.textContent = device.name;
    }
    
    // 更新设备显示（包括IP地址、网关等信息）
    updateDeviceIPDisplay(device);
    
    // 如果是WiFi路由器，更新所有使用相同SSID的笔记本的DHCP配置
    if (device.type === 'wireless-router' && device.properties.ssid) {
        updateLaptopsAfterRouterChange(device.properties.ssid);
    }
    
    // 记录日志
    log(`添加设备: ${device.name} (${device.type})`);
    
    // 选中新设备
    selectDevice(device);
    
    return device;
}

// 设备点击事件
function handleDeviceClick(deviceId) {
    const device = devices.find(d => d.id === deviceId);
    if (device) {
        if (isConnecting) {
            if (startDevice) {
                // 完成连接
                const connection = connectDevices(startDevice, device);
                // 如果连接成功，退出连接模式
                if (connection) {
                    isConnecting = false;
                    startDevice = null;
                    document.getElementById('connect-devices').classList.remove('active');
                    document.getElementById('connect-devices').style.backgroundColor = '#3498db';
                }
                // 如果连接失败，保持在连接模式，用户可以继续选择其他设备
            } else {
                // 设置起始设备
                startDevice = device;
                log(`选择起始设备: ${device.name}`);
            }
        } else {
            // 选择设备
            selectDevice(device);
        }
    }
}

// 连接设备
function connectDevices(device1, device2, autoSelect = true) {
    // 仿真运行时不能创建连接
    if (isSimulating) {
        alert('仿真运行时不能创建连接，请先停止仿真');
        return;
    }
    
    // 检查是否为同一个设备
    if (device1 === device2) {
        log("不能连接同一个设备");
        return;
    }
    
    // 检查是否已存在连接
    const existingConnection = connections.find(conn => 
        (conn.device1 === device1 && conn.device2 === device2) || 
        (conn.device1 === device2 && conn.device2 === device1)
    );
    
    if (existingConnection) {
        log("设备间已存在连接");
        return;
    }
    
    // 检查设备连接数量限制
    const checkDeviceConnectionLimit = (device) => {
        // 计算机和笔记本电脑只有一个接口，限制只能有一个连接
        if (device.type === 'computer' || device.type === 'laptop') {
            const currentConnections = device.connections.length;
            if (currentConnections >= 1) {
                log(`${device.name} 只有一个网络接口，无法创建更多连接`);
                return false;
            }
        }
        // 路由器和无线路由器限制连接数量不超过接口数量
        else if (device.type === 'router' || device.type === 'wireless-router') {
            const interfaceCount = device.properties.interfaces.length;
            const currentConnections = device.connections.length;
            if (currentConnections >= interfaceCount) {
                log(`${device.name} 的接口数量已达上限 (${interfaceCount}个接口)，无法创建新连接`);
                return false;
            }
        }
        // 交换机限制连接数量不超过端口数量
        else if (device.type === 'switch') {
            const portCount = device.properties.ports || 8;
            const currentConnections = device.connections.length;
            if (currentConnections >= portCount) {
                log(`${device.name} 的端口数量已达上限 (${portCount}个端口)，无法创建新连接`);
                return false;
            }
        }
        // 其他设备类型不限制连接数量
        return true;
    };
    
    if (!checkDeviceConnectionLimit(device1) || !checkDeviceConnectionLimit(device2)) {
        return;
    }
    
    const connectionId = `connection_${++connectionCounter}`;
    
    // 计算连接点
    const x1 = device1.x + device1.width / 2;
    const y1 = device1.y + device1.height / 2;
    const x2 = device2.x + device2.width / 2;
    const y2 = device2.y + device2.height / 2;
    
    // 创建连接线
    const line = document.createElementNS("http://www.w3.org/2000/svg", "line");
    line.setAttribute("x1", x1);
    line.setAttribute("y1", y1);
    line.setAttribute("x2", x2);
    line.setAttribute("y2", y2);
    line.setAttribute("class", "connection");
    line.setAttribute("id", connectionId);
    canvas.appendChild(line);
    
    // 创建连接点
    const dot1 = document.createElementNS("http://www.w3.org/2000/svg", "circle");
    dot1.setAttribute("cx", x1);
    dot1.setAttribute("cy", y1);
    dot1.setAttribute("r", "4");
    dot1.setAttribute("class", "connection-dot");
    canvas.appendChild(dot1);
    
    const dot2 = document.createElementNS("http://www.w3.org/2000/svg", "circle");
    dot2.setAttribute("cx", x2);
    dot2.setAttribute("cy", y2);
    dot2.setAttribute("r", "4");
    dot2.setAttribute("class", "connection-dot");
    canvas.appendChild(dot2);
    
    // 添加点击事件
    line.addEventListener('click', function(e) {
        e.stopPropagation();
        handleConnectionClick(connectionId);
    });
    
    // 创建连接对象
    const connection = {
        id: connectionId,
        device1: device1,
        device2: device2,
        x1: x1,
        y1: y1,
        x2: x2,
        y2: y2,
        element: line,
        dots: [dot1, dot2],
        isActive: true
    };
    
    try {
        // 添加到连接列表
        connections.push(connection);
        
        // 更新设备连接列表
        device1.connections.push(connection);
        device2.connections.push(connection);
        
        // 记录日志
        log(`创建连接: ${device1.name} ↔ ${device2.name}`);
        
        // 选中新连接（根据参数决定）
        if (autoSelect) {
            selectConnection(connection);
        }
        
        return connection;
    } catch (error) {
        // 清理创建的元素
        canvas.removeChild(line);
        canvas.removeChild(dot1);
        canvas.removeChild(dot2);
        
        // 记录错误
        log(`创建连接失败: ${error.message}`);
        console.error('创建连接失败:', error);
        
        return null;
    }
}

// 连接点击事件
function handleConnectionClick(connectionId) {
    const connection = connections.find(conn => conn.id === connectionId);
    if (connection) {
        selectConnection(connection);
    }
}

// 选择设备
function selectDevice(device) {
    // 取消之前的选择
    if (selectedDevice) {
        selectedDevice.element.classList.remove('selected');
    }
    if (selectedConnection) {
        selectedConnection.element.classList.remove('selected');
        selectedConnection = null;
    }
    
    // 选择新设备
    selectedDevice = device;
    selectedDevice.element.classList.add('selected');
    
    // 更新属性面板
    updateDeviceProperties(device);
    
    log(`选中设备: ${device.name}`);
}

// 选择连接
function selectConnection(connection) {
    // 取消所有连接的选中状态
    connections.forEach(conn => {
        conn.element.classList.remove('selected');
    });
    
    // 取消设备的选中状态
    if (selectedDevice) {
        selectedDevice.element.classList.remove('selected');
        selectedDevice = null;
    }
    
    // 选择新连接
    selectedConnection = connection;
    selectedConnection.element.classList.add('selected');
    
    // 更新属性面板
    updateConnectionProperties(connection);
    
    log(`选中连接: ${connection.device1.name} ↔ ${connection.device2.name}`);
}

// 更新设备属性面板
function updateDeviceProperties(device) {
    const content = `
        <h4>${device.name} 属性</h4>
        <div class="property-form">
            <div class="form-group">
                <label for="device-name">设备名称</label>
                <input type="text" id="device-name" value="${device.name}">
            </div>
            <div class="form-group">
                <label for="device-type">设备类型</label>
                <input type="text" id="device-type" value="${DEVICE_TYPES[device.type].name}" disabled>
            </div>
            
            <!-- 根据设备类型显示不同的属性 -->
            ${renderDeviceSpecificProperties(device)}
            
            <div class="form-actions">
                ${(device.type === 'computer' || device.type === 'laptop' || device.type === 'server' || device.type === 'dns-server' || device.type === 'web-server' || device.type === 'email-server' || device.type === 'switch' || device.type === 'wireless-router') ? `
                    <div style="display: flex; gap: 10px; width: 100%;">
                        <button class="btn-save" onclick="saveDeviceProperties()" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">保存配置</button>
                        <button class="btn-cancel" onclick="deleteDevice('${device.id}')" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">删除设备</button>
                    </div>
                ` : `
                    <button class="btn-cancel" onclick="deleteDevice('${device.id}')" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; width: 100%;">删除设备</button>
                `}
            </div>
        </div>
    `;
    
    document.getElementById('device-properties').innerHTML = `<h3>设备属性</h3><div class="properties-content">${content}</div>`;
    
    // 添加未保存更改提示功能
    setupUnsavedChangesIndicator(device);
    
    // 如果是笔记本，添加IP模式切换的即时更新事件
    if (device.type === 'laptop') {
        const ipModeSelect = document.getElementById('ip-mode');
        if (ipModeSelect) {
            ipModeSelect.addEventListener('change', function() {
                const staticConfig = document.getElementById('static-ip-config');
                
                if (this.value === 'static') {
                    // 静态模式：显示IP配置区域
                    if (staticConfig) {
                        staticConfig.style.display = 'block';
                    }
                    // 如果IP为空，使用默认值
                    if (!device.properties.ip) {
                        device.properties.ip = '192.168.1.101';
                        device.properties.subnetMask = '255.255.255.0';
                        device.properties.gateway = '192.168.1.1';
                        device.properties.dns = '8.8.8.8';
                    }
                } else {
                    // DHCP模式：隐藏IP配置区域
                    if (staticConfig) {
                        staticConfig.style.display = 'none';
                    }
                    // 尝试自动配置
                    if (typeof handleAutomaticNetworkConfig === 'function') {
                        handleAutomaticNetworkConfig();
                    }
                }
                
                // 更新设备的ipMode属性
                device.properties.ipMode = this.value;
                
                // 立即更新设备图标下方的显示
                if (typeof updateDeviceIPDisplay === 'function') {
                    updateDeviceIPDisplay(device);
                }
            });
        }
    }
}

// 设置未保存更改提示
function setupUnsavedChangesIndicator(device) {
    // 保存原始值用于比较
    const originalValues = {};
    
    // 获取所有输入字段（包括动态添加的）
    function getAllInputs() {
        return document.querySelectorAll('#device-properties input, #device-properties select, #device-properties textarea');
    }
    
    // 记录所有输入字段的原始值
    function recordOriginalValues() {
        const inputs = getAllInputs();
        inputs.forEach(input => {
            if (input.id && !input.disabled) {
                originalValues[input.id] = input.value;
            }
        });
    }
    
    // 初始记录
    recordOriginalValues();
    
    // 添加变化监听器（使用事件委托，支持动态添加的字段）
    const propertiesContent = document.querySelector('#device-properties .properties-content');
    if (propertiesContent) {
        propertiesContent.addEventListener('input', function(e) {
            if (e.target.matches('input, select, textarea') && !e.target.disabled) {
                checkForUnsavedChanges();
            }
        });
        propertiesContent.addEventListener('change', function(e) {
            if (e.target.matches('input, select, textarea') && !e.target.disabled) {
                checkForUnsavedChanges();
            }
        });
    }
    
    // 检查是否有未保存的更改
    function checkForUnsavedChanges() {
        let hasChanges = false;
        const inputs = getAllInputs();
        
        inputs.forEach(input => {
            if (input.id && !input.disabled) {
                const currentValue = input.value;
                const originalValue = originalValues[input.id];
                
                // 如果字段不在原始值中，说明是新添加的字段，需要重新记录
                if (originalValue === undefined) {
                    originalValues[input.id] = currentValue;
                } else if (currentValue !== originalValue) {
                    hasChanges = true;
                }
            }
        });
        
        // 更新标题显示
        const titleElement = document.querySelector('#device-properties h3');
        if (titleElement) {
            if (hasChanges) {
                if (!titleElement.textContent.includes('*')) {
                    titleElement.textContent = '设备属性 *';
                    titleElement.style.color = '#e74c3c';
                }
            } else {
                if (titleElement.textContent.includes('*')) {
                    titleElement.textContent = '设备属性';
                    titleElement.style.color = '';
                }
            }
        }
    }
}

// 切换密码显示/隐藏
function togglePasswordVisibility(inputId) {
    const passwordInput = document.getElementById(inputId);
    if (!passwordInput) return;
    
    // 查找图标元素（可能在input的父容器中）
    const wrapper = passwordInput.closest('.password-input-wrapper');
    const toggleIcon = wrapper ? wrapper.querySelector('.password-toggle-icon') : passwordInput.nextElementSibling;
    
    if (toggleIcon) {
        if (passwordInput.type === 'password') {
            passwordInput.type = 'text';
            toggleIcon.textContent = '◉'; // 带点的实心圆，表示密码可见
            toggleIcon.title = '隐藏密码';
        } else {
            passwordInput.type = 'password';
            toggleIcon.textContent = '◎'; // 带点的空心圆，表示密码隐藏
            toggleIcon.title = '显示密码';
        }
    }
}

// 渲染设备特定属性
function renderDeviceSpecificProperties(device) {
    let html = '';
    
    switch (device.type) {
        case 'computer':
        case 'server':
            html = `
                <div class="form-group">
                    <label for="device-ip">IP地址</label>
                    <input type="text" id="device-ip" value="${device.properties.ip}">
                </div>
                <div class="form-group">
                    <label for="device-subnet">子网掩码</label>
                    <input type="text" id="device-subnet" value="${device.properties.subnetMask}">
                </div>
                <div class="form-group">
                    <label for="device-gateway">默认网关</label>
                    <input type="text" id="device-gateway" value="${device.properties.gateway}">
                </div>
                <div class="form-group">
                    <label for="device-dns">DNS服务器</label>
                    <input type="text" id="device-dns" value="${device.properties.dns}">
                </div>
            `;
            break;
            
        case 'laptop':
            html = `
                <div class="form-group">
                    <label for="device-ssid">无线SSID</label>
                    <input type="text" id="device-ssid" value="${device.properties.ssid}">
                </div>
                <div class="form-group">
                    <label for="device-password">无线密码</label>
                    <div class="password-input-wrapper">
                        <input type="password" id="device-password" value="${device.properties.password}">
                        <span class="password-toggle-icon" onclick="togglePasswordVisibility('device-password')" title="显示/隐藏密码">◎</span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="ip-mode">IP获取方式</label>
                    <select id="ip-mode">
                        <option value="dhcp" ${device.properties.ipMode === 'dhcp' ? 'selected' : ''}>动态分配(DHCP)</option>
                        <option value="static" ${device.properties.ipMode === 'static' ? 'selected' : ''}>静态分配</option>
                    </select>
                </div>
                <div id="static-ip-config" ${device.properties.ipMode === 'dhcp' ? 'style="display:none;"' : ''}>
                    <div class="form-group">
                        <label for="device-ip">IP地址</label>
                        <input type="text" id="device-ip" value="${device.properties.ip || ''}">
                    </div>
                    <div class="form-group">
                        <label for="device-subnet">子网掩码</label>
                        <input type="text" id="device-subnet" value="${device.properties.subnetMask || ''}">
                    </div>
                    <div class="form-group">
                        <label for="device-gateway">默认网关</label>
                        <input type="text" id="device-gateway" value="${device.properties.gateway || ''}">
                    </div>
                    <div class="form-group">
                        <label for="device-dns">DNS服务器</label>
                        <input type="text" id="device-dns" value="${device.properties.dns || ''}">
                    </div>
                </div>

            `;
            break;
            
        case 'wireless-router':
            html = `
                <div class="form-group">
                    <label>网络接口</label>
                    ${device.properties.interfaces.map((iface, index) => `
                        <div style="margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px; display: flex; flex-direction: column; gap: 5px;">
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <h5 style="margin: 0;font-size: 14px;">接口 ${index + 1} (${iface.name})</h5>
                                <button type="button" onclick="removeInterface('${device.id}', ${index})" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 3px 8px; font-size: 12px; cursor: pointer;">
                                    删除
                                </button>
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="iface-name-${device.id}-${index}">接口名称</label>
                                <input type="text" id="iface-name-${device.id}-${index}" value="${iface.name}">
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="iface-ip-${device.id}-${index}">IP地址</label>
                                <input type="text" id="iface-ip-${device.id}-${index}" value="${iface.ip}">
                            </div>
                            <div class="form-group">
                                <label for="iface-subnet-${device.id}-${index}">子网掩码</label>
                                <input type="text" id="iface-subnet-${device.id}-${index}" value="${iface.subnetMask}">
                            </div>
                        </div>
                    `).join('')}
                    <div style="display: flex; gap: 10px; margin-top: 10px;">
                        <button type="button" onclick="addInterface('${device.id}')" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">
                            添加接口
                        </button>
                    </div>
                </div>
                <div class="form-group" style="margin-top: 20px;">
                    <label>无线设置 (关联内网接口: wlan0)</label>
                    <div class="form-group">
                        <label for="wireless-ssid-${device.id}">SSID</label>
                        <input type="text" id="wireless-ssid-${device.id}" value="${device.properties.ssid}">
                    </div>
                    <div class="form-group">
                        <label for="wireless-security-${device.id}">加密方式</label>
                        <select id="wireless-security-${device.id}">
                            <option value="open" ${device.properties.security === 'open' ? 'selected' : ''}>开放</option>
                            <option value="WPA2-PSK" ${device.properties.security === 'WPA2-PSK' ? 'selected' : ''}>WPA2-PSK</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="wireless-password-${device.id}">密码</label>
                        <div class="password-input-wrapper">
                            <input type="password" id="wireless-password-${device.id}" value="${device.properties.password}">
                            <span class="password-toggle-icon" onclick="togglePasswordVisibility('wireless-password-${device.id}')" title="显示/隐藏密码">◎</span>
                        </div>
                    </div>
                </div>
                <div class="form-group" style="margin-top: 20px;">
                    <label>DHCP服务器设置</label>
                    <div class="form-group">
                        <label for="dhcp-dns-${device.id}">DNS服务器地址</label>
                        <input type="text" id="dhcp-dns-${device.id}" value="${device.properties.dhcpDns || '8.8.8.8'}">
                    </div>
                </div>
                <div class="form-group" style="margin-top: 20px;">
                    <label>路由表</label>
                    <div style="max-height: 300px; overflow-y: auto;">
                        ${device.properties.routingTable.map((route, index) => `
                            <div style="margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px; display: flex; flex-direction: column; gap: 5px;">
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <h5 style="margin: 0;font-size: 14px;">路由 ${index + 1}</h5>
                                    <button type="button" onclick="removeRoute('${device.id}', ${index})" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 3px 8px; font-size: 12px; cursor: pointer;">
                                        删除
                                    </button>
                                </div>
                                <div class="form-group" style="margin-bottom: 5px;">
                                    <label for="route-network-${device.id}-${index}">目标网络</label>
                                    <input type="text" id="route-network-${device.id}-${index}" value="${route.network}">
                                </div>
                                <div class="form-group" style="margin-bottom: 5px;">
                                    <label for="route-mask-${device.id}-${index}">子网掩码</label>
                                    <input type="text" id="route-mask-${device.id}-${index}" value="${route.subnetMask}">
                                </div>
                                <div class="form-group">
                                    <label for="route-next-hop-${device.id}-${index}">下一跳IP</label>
                                    <input type="text" id="route-next-hop-${device.id}-${index}" value="${route.nextHop}">
                                </div>
                            </div>
                        `).join('')}
                    </div>
                    <div style="display: flex; gap: 10px; margin-top: 10px;">
                        <button type="button" onclick="addRoute('${device.id}')" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">
                            添加静态路由
                        </button>
                    </div>
                </div>
            `;
            break;
        case 'router':
            html = html + `
                <div class="form-group">
                    <label>网络接口</label>
                    ${device.properties.interfaces.map((iface, index) => `
                        <div style="margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px; display: flex; flex-direction: column; gap: 5px;">
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <h5 style="margin: 0;font-size: 14px;">接口 ${index + 1}</h5>
                                <button type="button" onclick="removeInterface('${device.id}', ${index})" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 3px 8px; font-size: 12px; cursor: pointer;">
                                    删除
                                </button>
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="iface-name-${device.id}-${index}">接口名称</label>
                                <input type="text" id="iface-name-${device.id}-${index}" value="${iface.name}">
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="iface-ip-${device.id}-${index}">IP地址</label>
                                <input type="text" id="iface-ip-${device.id}-${index}" value="${iface.ip}">
                            </div>
                            <div class="form-group">
                                <label for="iface-subnet-${device.id}-${index}">子网掩码</label>
                                <input type="text" id="iface-subnet-${device.id}-${index}" value="${iface.subnetMask}">
                            </div>
                        </div>
                    `).join('')}
                    <div style="display: flex; gap: 10px; margin-top: 10px;">
                        <button type="button" onclick="addInterface('${device.id}')" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">
                            添加接口
                        </button>
                        <button class="btn-save" onclick="saveRouterInterfaces()" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">保存接口配置</button>
                    </div>
                </div>
                <div class="form-group" style="margin-top: 20px;">
                    <label>静态路由表</label>
                    ${device.properties.routingTable.map((route, index) => `
                        <div style="margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px; display: flex; flex-direction: column; gap: 5px;">
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <h5 style="margin: 0;font-size: 14px;">路由 ${index + 1}</h5>
                                <button type="button" onclick="removeRoute('${device.id}', ${index})" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 3px 8px; font-size: 12px; cursor: pointer;">
                                    删除
                                </button>
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="route-network-${device.id}-${index}">目标网络</label>
                                <input type="text" id="route-network-${device.id}-${index}" value="${route.network}">
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="route-mask-${device.id}-${index}">子网掩码</label>
                                <input type="text" id="route-mask-${device.id}-${index}" value="${route.subnetMask}">
                            </div>
                            <div class="form-group">
                                <label for="route-next-hop-${device.id}-${index}">下一跳IP</label>
                                <input type="text" id="route-next-hop-${device.id}-${index}" value="${route.nextHop}">
                            </div>
                        </div>
                    `).join('')}
                    <div style="display: flex; gap: 10px; margin-top: 10px;">
                        <button type="button" onclick="addRoute('${device.id}')" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">
                            添加静态路由
                        </button>
                        <button class="btn-save" onclick="saveRouterRoutes()" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; cursor: pointer; flex: 1;">保存路由表</button>
                    </div>
                </div>
            `;
            break;
            
        case 'switch':
            html = `
                <div class="form-group">
                    <label for="device-ports">端口数量</label>
                    <input type="number" id="device-ports" value="${device.properties.ports}" min="1" max="24">
                </div>
            `;
            break;
            
        case 'dns-server':
            html = `
                <div class="form-group">
                    <label for="device-ip">IP地址</label>
                    <input type="text" id="device-ip" value="${device.properties.ip}">
                </div>
                <div class="form-group">
                    <label for="device-subnet">子网掩码</label>
                    <input type="text" id="device-subnet" value="${device.properties.subnetMask}">
                </div>
                <div class="form-group">
                    <label for="device-gateway">默认网关</label>
                    <input type="text" id="device-gateway" value="${device.properties.gateway || ''}">
                </div>
                <div class="form-group">
                    <label for="device-dns">DNS服务器</label>
                    <input type="text" id="device-dns" value="${device.properties.dns || ''}">
                </div>
                <div class="form-group">
                    <label>域名映射</label>
                    ${device.properties.domains.map((domain, index) => `
                        <div style="margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px; display: flex; flex-direction: column; gap: 5px;">
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <h5 style="margin: 0;font-size: 14px;">域名 ${index + 1}</h5>
                                <button type="button" onclick="removeDomain('${device.id}', ${index})" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 3px 8px; font-size: 12px; cursor: pointer;">
                                    删除
                                </button>
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="domain-name-${device.id}-${index}">域名</label>
                                <input type="text" id="domain-name-${device.id}-${index}" value="${domain.name}">
                            </div>
                            <div class="form-group" style="margin-bottom: 5px;">
                                <label for="domain-type-${device.id}-${index}">记录类型</label>
                                <select id="domain-type-${device.id}-${index}">
                                    <option value="A" ${domain.type === 'A' ? 'selected' : ''}>A记录</option>
                                    <option value="AAAA" ${domain.type === 'AAAA' ? 'selected' : ''}>AAAA记录</option>
                                    <option value="CNAME" ${domain.type === 'CNAME' ? 'selected' : ''}>CNAME记录</option>
                                    <option value="MX" ${domain.type === 'MX' ? 'selected' : ''}>MX记录</option>
                                    <option value="NS" ${domain.type === 'NS' ? 'selected' : ''}>NS记录</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="domain-ip-${device.id}-${index}">
                                    ${domain.type === 'MX' ? '邮件服务器IP' : 
                                      domain.type === 'CNAME' ? '别名' : 
                                      domain.type === 'NS' ? '名称服务器' : 
                                      'IP地址'}
                                </label>
                                <input type="text" id="domain-ip-${device.id}-${index}" value="${domain.ip}">
                            </div>
                        </div>
                    `).join('')}
                    <button type="button" onclick="addDomain('${device.id}')" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; margin-top: 10px; cursor: pointer;">
                        添加域名
                    </button>
                </div>
            `;
            break;
            
        case 'web-server':
            html = `
                <div class="form-group">
                    <label for="device-ip">IP地址</label>
                    <input type="text" id="device-ip" value="${device.properties.ip}">
                </div>
                <div class="form-group">
                    <label for="device-subnet">子网掩码</label>
                    <input type="text" id="device-subnet" value="${device.properties.subnetMask}">
                </div>
                <div class="form-group">
                    <label for="device-gateway">默认网关</label>
                    <input type="text" id="device-gateway" value="${device.properties.gateway || ''}">
                </div>
                <div class="form-group">
                    <label for="device-dns">DNS服务器</label>
                    <input type="text" id="device-dns" value="${device.properties.dns || ''}">
                </div>
                <div class="form-group">
                    <label for="website-title">网站标题</label>
                    <input type="text" id="website-title" value="${device.properties.website.title}">
                </div>
                <div class="form-group">
                    <label for="website-content">网站内容</label>
                    <textarea id="website-content" rows="4" style="width: 100%;">${device.properties.website.content}</textarea>
                </div>
            `;
            break;
            
        case 'email-server':
            html = `
                <div class="form-group">
                    <label for="device-ip">IP地址</label>
                    <input type="text" id="device-ip" value="${device.properties.ip}">
                </div>
                <div class="form-group">
                    <label for="device-subnet">子网掩码</label>
                    <input type="text" id="device-subnet" value="${device.properties.subnetMask}">
                </div>
                <div class="form-group">
                    <label for="device-gateway">默认网关</label>
                    <input type="text" id="device-gateway" value="${device.properties.gateway || ''}">
                </div>
                <div class="form-group">
                    <label for="device-dns">DNS服务器</label>
                    <input type="text" id="device-dns" value="${device.properties.dns || ''}">
                </div>
                <div class="form-group">
                    <label for="device-domain">邮件域名</label>
                    <input type="text" id="device-domain" value="${device.properties.domain}">
                </div>
                <div class="form-group">
                    <label for="smtp-port">SMTP端口</label>
                    <input type="number" id="smtp-port" value="${device.properties.smtpPort}" min="1" max="65535">
                </div>
                <div class="form-group">
                    <label for="pop3-port">POP3端口</label>
                    <input type="number" id="pop3-port" value="${device.properties.pop3Port}" min="1" max="65535">
                </div>
                <div class="form-group">
                    <label for="imap-port">IMAP端口</label>
                    <input type="number" id="imap-port" value="${device.properties.imapPort}" min="1" max="65535">
                </div>
                <div class="form-group">
                    <label>邮箱用户</label>
                    ${device.properties.users.map((user, index) => `
                        <div style="margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px; display: flex; align-items: center; gap: 10px;">
                            <div style="flex: 1; min-width: 120px;">
                                <label for="user-username-${device.id}-${index}">用户名</label>
                                <input type="text" id="user-username-${device.id}-${index}" value="${user.username}" style="width: 120px;">
                            </div>
                            <div style="flex: 1; min-width: 120px;">
                                <label for="user-password-${device.id}-${index}">密码</label>
                                <div class="password-input-wrapper" style="position: relative;">
                                    <input type="password" id="user-password-${device.id}-${index}" value="${user.password}" style="width: 120px; padding-right: 30px;">
                                    <span class="password-toggle-icon" onclick="togglePasswordVisibility('user-password-${device.id}-${index}')" title="显示/隐藏密码" style="right: 5px;">◎</span>
                                </div>
                            </div>
                            <button type="button" onclick="removeEmailUser('${device.id}', ${index})" style="background-color: #e74c3c; color: white; border: none; border-radius: 4px; padding: 4px 8px; cursor: pointer;">
                                删除
                            </button>
                        </div>
                    `).join('')}
                    <button type="button" onclick="addEmailUser('${device.id}')" style="background-color: #27ae60; color: white; border: none; border-radius: 4px; padding: 8px 12px; margin-top: 10px; cursor: pointer;">
                        添加用户
                    </button>
                </div>
            `;
            break;
    }
    
    return html;
}

// 更新连接属性面板
function updateConnectionProperties(connection) {
    const content = `
        <h4>连接属性</h4>
        <div class="property-form">
            <div class="form-group">
                <label>连接设备</label>
                <input type="text" value="${connection.device1.name} ↔ ${connection.device2.name}" disabled>
            </div>
            <div class="form-group">
                <label for="connection-status">连接状态</label>
                <select id="connection-status">
                    <option value="active" ${connection.isActive ? 'selected' : ''}>激活</option>
                    <option value="inactive" ${!connection.isActive ? 'selected' : ''}>未激活</option>
                </select>
            </div>
            <div class="form-actions">
                <button class="btn-save" onclick="saveConnectionProperties()">保存</button>
                <button class="btn-cancel" onclick="cancelProperties()">取消</button>
                <button class="btn-cancel" onclick="deleteConnection()" style="background-color: #e74c3c;">删除</button>
            </div>
        </div>
    `;
    
    document.getElementById('connection-properties').innerHTML = `<h3>连接属性</h3><div class="properties-content">${content}</div>`;
}

// 保存设备属性
function saveDeviceProperties() {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    if (!selectedDevice) return;
    
    // 更新基本属性
    selectedDevice.name = document.getElementById('device-name').value;
    
    // 更新设备特定属性
    switch (selectedDevice.type) {
        case 'computer':
            selectedDevice.properties.ip = document.getElementById('device-ip').value;
            selectedDevice.properties.subnetMask = document.getElementById('device-subnet').value;
            selectedDevice.properties.gateway = document.getElementById('device-gateway').value;
            selectedDevice.properties.dns = document.getElementById('device-dns').value;
            break;
            
        case 'router':
            // 更新接口信息
            selectedDevice.properties.interfaces.forEach((iface, index) => {
                iface.name = document.getElementById(`iface-name-${selectedDevice.id}-${index}`).value;
                iface.ip = document.getElementById(`iface-ip-${selectedDevice.id}-${index}`).value;
                iface.subnetMask = document.getElementById(`iface-subnet-${selectedDevice.id}-${index}`).value;
            });
            // 更新路由表信息
            selectedDevice.properties.routingTable.forEach((route, index) => {
                route.network = document.getElementById(`route-network-${selectedDevice.id}-${index}`).value;
                route.subnetMask = document.getElementById(`route-mask-${selectedDevice.id}-${index}`).value;
                route.nextHop = document.getElementById(`route-next-hop-${selectedDevice.id}-${index}`).value;
            });
            break;
            
        case 'wireless-router':
            // 记录旧的SSID，以便检查是否有变化
            const oldSSID = selectedDevice.properties.ssid;
            
            // 更新无线设置
            selectedDevice.properties.ssid = document.getElementById(`wireless-ssid-${selectedDevice.id}`).value;
            selectedDevice.properties.security = document.getElementById(`wireless-security-${selectedDevice.id}`).value;
            selectedDevice.properties.password = document.getElementById(`wireless-password-${selectedDevice.id}`).value;
            
            // 更新DHCP设置
            selectedDevice.properties.dhcpDns = document.getElementById(`dhcp-dns-${selectedDevice.id}`).value;
            
            // 更新接口信息
            selectedDevice.properties.interfaces.forEach((iface, index) => {
                iface.name = document.getElementById(`iface-name-${selectedDevice.id}-${index}`).value;
                iface.ip = document.getElementById(`iface-ip-${selectedDevice.id}-${index}`).value;
                iface.subnetMask = document.getElementById(`iface-subnet-${selectedDevice.id}-${index}`).value;
            });
            // 更新路由表信息
            selectedDevice.properties.routingTable.forEach((route, index) => {
                route.network = document.getElementById(`route-network-${selectedDevice.id}-${index}`).value;
                route.subnetMask = document.getElementById(`route-mask-${selectedDevice.id}-${index}`).value;
                route.nextHop = document.getElementById(`route-next-hop-${selectedDevice.id}-${index}`).value;
            });
            
            // 如果SSID或接口配置发生变化，更新所有使用DHCP模式的笔记本
            if (oldSSID !== selectedDevice.properties.ssid) {
                // SSID改变，需要更新所有使用旧SSID和新SSID的笔记本
                updateLaptopsAfterRouterChange(oldSSID);
                updateLaptopsAfterRouterChange(selectedDevice.properties.ssid);
            } else {
                // SSID未改变，但接口配置可能改变，更新使用此SSID的笔记本
                updateLaptopsAfterRouterChange(selectedDevice.properties.ssid);
            }
            break;
            
        case 'switch':
            selectedDevice.properties.ports = parseInt(document.getElementById('device-ports').value);
            break;
            
        case 'dns-server':
            selectedDevice.properties.ip = document.getElementById('device-ip').value;
            selectedDevice.properties.subnetMask = document.getElementById('device-subnet').value;
            selectedDevice.properties.gateway = document.getElementById('device-gateway').value;
            selectedDevice.properties.dns = document.getElementById('device-dns').value;
            selectedDevice.properties.domains.forEach((domain, index) => {
                domain.name = document.getElementById(`domain-name-${selectedDevice.id}-${index}`).value;
                domain.ip = document.getElementById(`domain-ip-${selectedDevice.id}-${index}`).value;
                domain.type = document.getElementById(`domain-type-${selectedDevice.id}-${index}`).value;
            });
            break;
            
        case 'web-server':
            selectedDevice.properties.ip = document.getElementById('device-ip').value;
            selectedDevice.properties.subnetMask = document.getElementById('device-subnet').value;
            selectedDevice.properties.gateway = document.getElementById('device-gateway').value;
            selectedDevice.properties.dns = document.getElementById('device-dns').value;
            selectedDevice.properties.website.title = document.getElementById('website-title').value;
            selectedDevice.properties.website.content = document.getElementById('website-content').value;
            break;
            
        case 'email-server':
            selectedDevice.properties.ip = document.getElementById('device-ip').value;
            selectedDevice.properties.subnetMask = document.getElementById('device-subnet').value;
            selectedDevice.properties.gateway = document.getElementById('device-gateway').value;
            selectedDevice.properties.dns = document.getElementById('device-dns').value;
            selectedDevice.properties.domain = document.getElementById('device-domain').value;
            selectedDevice.properties.smtpPort = parseInt(document.getElementById('smtp-port').value);
            selectedDevice.properties.pop3Port = parseInt(document.getElementById('pop3-port').value);
            selectedDevice.properties.imapPort = parseInt(document.getElementById('imap-port').value);
            selectedDevice.properties.users.forEach((user, index) => {
                user.username = document.getElementById(`user-username-${selectedDevice.id}-${index}`).value;
                user.password = document.getElementById(`user-password-${selectedDevice.id}-${index}`).value;
            });
            break;
            
        case 'laptop':
            // 保存无线配置
            selectedDevice.properties.ssid = document.getElementById('device-ssid').value;
            selectedDevice.properties.password = document.getElementById('device-password').value;
            selectedDevice.properties.ipMode = document.getElementById('ip-mode').value;
            
            // 根据IP模式保存不同的配置
            if (selectedDevice.properties.ipMode === 'static') {
                // 静态模式下保存所有配置
                selectedDevice.properties.ip = document.getElementById('device-ip').value;
                selectedDevice.properties.subnetMask = document.getElementById('device-subnet').value;
                selectedDevice.properties.gateway = document.getElementById('device-gateway').value;
                selectedDevice.properties.dns = document.getElementById('device-dns').value;
            } else {
                // DHCP模式下保存基本配置，IP、子网掩码、网关和DNS由自动配置分配
                // 不清空当前值，保留现有配置直到DHCP分配成功
                // 如果当前没有IP，则使用默认值
                const currentIP = selectedDevice.properties.ip;
                const currentSubnetMask = selectedDevice.properties.subnetMask;
                const currentGateway = selectedDevice.properties.gateway;
                const currentDNS = selectedDevice.properties.dns;
                
                // 调用自动网络配置函数（可能会更新IP）
                handleAutomaticNetworkConfig();
                
                // 如果DHCP分配失败（没有找到匹配的路由器），保留原有值或使用默认值
                if (!selectedDevice.properties.ip) {
                    selectedDevice.properties.ip = currentIP || '192.168.1.101';
                }
                if (!selectedDevice.properties.subnetMask) {
                    selectedDevice.properties.subnetMask = currentSubnetMask || '255.255.255.0';
                }
                if (!selectedDevice.properties.gateway) {
                    selectedDevice.properties.gateway = currentGateway || '192.168.1.1';
                }
                if (!selectedDevice.properties.dns) {
                    selectedDevice.properties.dns = currentDNS || '8.8.8.8';
                }
            }
            break;
    }
    
    // 更新设备标签
    const label = selectedDevice.element.querySelector('.device-label');
    if (label) {
        label.textContent = selectedDevice.name;
    }
    
    // 更新设备IP和网关显示
    updateDeviceIPDisplay(selectedDevice);
    
    log(`更新设备属性: ${selectedDevice.name}`);
    
    // 清除未保存提示
    const titleElement = document.querySelector('#device-properties h3');
    if (titleElement && titleElement.textContent.includes('*')) {
        titleElement.textContent = '设备属性';
        titleElement.style.color = '';
    }
    
    // 重新选择设备以更新属性面板
    selectDevice(selectedDevice);
    
    // 更新WiFi连接虚线
    updateWirelessConnections();
    
    // 处理DHCP配置
    handleAutomaticNetworkConfig();
}

// 保存路由器接口配置
function saveRouterInterfaces() {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    if (!selectedDevice) return;
    if (selectedDevice.type !== 'router' && selectedDevice.type !== 'wireless-router') return;
    
    // 更新基本属性（名称）
    selectedDevice.name = document.getElementById('device-name').value;
    
    // 更新接口信息
    selectedDevice.properties.interfaces.forEach((iface, index) => {
        iface.name = document.getElementById(`iface-name-${selectedDevice.id}-${index}`).value;
        iface.ip = document.getElementById(`iface-ip-${selectedDevice.id}-${index}`).value;
        iface.subnetMask = document.getElementById(`iface-subnet-${selectedDevice.id}-${index}`).value;
    });
    
    // 更新设备标签
    const label = selectedDevice.element.querySelector('.device-label');
    if (label) {
        label.textContent = selectedDevice.name;
    }
    
    // 更新设备IP和网关显示
    updateDeviceIPDisplay(selectedDevice);
    
    log(`更新路由器接口配置: ${selectedDevice.name}`);
    
    // 如果是WiFi路由器，更新所有使用DHCP模式的笔记本
    if (selectedDevice.type === 'wireless-router') {
        updateLaptopsAfterRouterChange(selectedDevice.properties.ssid);
    }
    
    // 清除未保存提示
    const titleElement = document.querySelector('#device-properties h3');
    if (titleElement && titleElement.textContent.includes('*')) {
        titleElement.textContent = '设备属性';
        titleElement.style.color = '';
    }
    
    // 重新选择设备以更新属性面板
    selectDevice(selectedDevice);
    
    // 更新WiFi连接虚线
    updateWirelessConnections();
    
    // 处理DHCP配置
    handleAutomaticNetworkConfig();
}

// 保存路由器路由表
function saveRouterRoutes() {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    if (!selectedDevice) return;
    if (selectedDevice.type !== 'router' && selectedDevice.type !== 'wireless-router') return;
    
    // 更新路由表信息
    selectedDevice.properties.routingTable.forEach((route, index) => {
        route.network = document.getElementById(`route-network-${selectedDevice.id}-${index}`).value;
        route.subnetMask = document.getElementById(`route-mask-${selectedDevice.id}-${index}`).value;
        route.nextHop = document.getElementById(`route-next-hop-${selectedDevice.id}-${index}`).value;
    });
    
    // 更新设备IP和网关显示
    updateDeviceIPDisplay(selectedDevice);
    
    log(`更新路由器路由表: ${selectedDevice.name}`);
    
    // 如果是WiFi路由器，更新所有使用DHCP模式的笔记本
    if (selectedDevice.type === 'wireless-router') {
        updateLaptopsAfterRouterChange(selectedDevice.properties.ssid);
    }
    
    // 清除未保存提示
    const titleElement = document.querySelector('#device-properties h3');
    if (titleElement && titleElement.textContent.includes('*')) {
        titleElement.textContent = '设备属性';
        titleElement.style.color = '';
    }
    
    // 重新选择设备以更新属性面板
    selectDevice(selectedDevice);
    
    // 更新WiFi连接虚线
    updateWirelessConnections();
    
    // 处理DHCP配置
    handleAutomaticNetworkConfig();
}

// 绘制或更新WiFi连接虚线
function updateWirelessConnections() {
    // 先移除所有现有的WiFi虚线连接
    const existingWirelessLines = document.querySelectorAll('.wireless-connection');
    existingWirelessLines.forEach(line => {
        canvas.removeChild(line);
    });
    
    // 遍历所有笔记本电脑
    devices.forEach(device => {
        if (device.type === 'laptop') {
            // 检查是否设置了SSID
            if (device.properties.ssid) {
                // 查找匹配SSID的无线路由器
                const wirelessRouter = devices.find(router => {
                    return router.type === 'wireless-router' && router.properties.ssid === device.properties.ssid;
                });
                
                // 检查密码是否正确
                let passwordMatch = true;
                if (wirelessRouter) {
                    // 如果路由器启用了加密，检查密码
                    if (wirelessRouter.properties.security && wirelessRouter.properties.security !== 'open') {
                        passwordMatch = (device.properties.password === wirelessRouter.properties.password);
                    }
                }
                
                if (wirelessRouter && passwordMatch) {
                    // 找到无线路由器的无线接口IP地址
                    const wirelessInterface = wirelessRouter.properties.interfaces.find(iface => iface.name.startsWith('wlan'));
                    if (wirelessInterface) {
                        // 检查笔记本电脑和无线路由器的无线接口是否在同一个网段
                        const laptopIp = device.properties.ip;
                        const laptopMask = device.properties.subnetMask;
                        const routerIp = wirelessInterface.ip;
                        const routerMask = wirelessInterface.subnetMask;
                        
                        if (isSameSubnet(laptopIp, laptopMask, routerIp, routerMask)) {
                            // 计算连接点
                            const x1 = device.x + device.width / 2;
                            const y1 = device.y + device.height / 2;
                            const x2 = wirelessRouter.x + wirelessRouter.width / 2;
                            const y2 = wirelessRouter.y + wirelessRouter.height / 2;
                            
                            // 创建虚线连接线
                            const line = document.createElementNS("http://www.w3.org/2000/svg", "line");
                            line.setAttribute("x1", x1);
                            line.setAttribute("y1", y1);
                            line.setAttribute("x2", x2);
                            line.setAttribute("y2", y2);
                            line.setAttribute("class", "wireless-connection");
                            line.setAttribute("id", `wireless_${device.id}_${wirelessRouter.id}`);
                            line.setAttribute("stroke", "#3498db");
                            line.setAttribute("stroke-width", "2");
                            line.setAttribute("stroke-dasharray", "5,5");
                            canvas.appendChild(line);
                        }
                    }
                }
            }
        }
    });
}

// 更新设备IP和网关显示
function updateDeviceIPDisplay(device) {
    // 移除旧的显示元素
    const oldIpText = device.element.querySelector('.device-ip');
    const oldGatewayText = device.element.querySelector('.device-gateway');
    const oldDnsText = device.element.querySelector('.device-dns');
    const oldDomainText = device.element.querySelector('.device-domain');
    const oldInterfaceTexts = device.element.querySelectorAll('.device-interface');
    const oldRoutingTableTexts = device.element.querySelectorAll('.device-routing-table');
    
    if (oldIpText) {
        device.element.removeChild(oldIpText);
    }
    if (oldGatewayText) {
        device.element.removeChild(oldGatewayText);
    }
    if (oldDnsText) {
        device.element.removeChild(oldDnsText);
    }
    if (oldDomainText) {
        device.element.removeChild(oldDomainText);
    }
    oldInterfaceTexts.forEach(text => {
        device.element.removeChild(text);
    });
    oldRoutingTableTexts.forEach(text => {
        device.element.removeChild(text);
    });
    
    // 根据设备类型处理不同的显示
    const deviceData = DEVICE_TYPES[device.type];
    
    // 检查设备类型是否存在
    if (!deviceData) {
        log(`警告：未知设备类型 ${device.type}，无法更新IP显示`);
        return;
    }
    
    if (device.type === 'router' || device.type === 'wireless-router') {
        // 显示所有接口的IP地址
        let yPosition = deviceData.height + 25;
        
        device.properties.interfaces.forEach((iface, index) => {
            if (!iface) return;
            const interfaceText = document.createElementNS("http://www.w3.org/2000/svg", "text");
            interfaceText.textContent = `${iface.name || '未知接口'}: ${iface.ip || ''}/${getMaskLength(iface.subnetMask)}`;
            interfaceText.setAttribute("class", "device-interface");
            interfaceText.setAttribute("x", deviceData.width / 2);
            interfaceText.setAttribute("y", yPosition);
            interfaceText.setAttribute("text-anchor", "middle");
            interfaceText.setAttribute("font-size", "10");
            interfaceText.setAttribute("fill", "#7f8c8d");
            device.element.appendChild(interfaceText);
            
            yPosition += 15;
        });
        
        // 显示路由表信息
        if (device.properties.routingTable.length > 0) {
            // 路由表标题
            const routingTableTitle = document.createElementNS("http://www.w3.org/2000/svg", "text");
            routingTableTitle.textContent = "路由表:";
            routingTableTitle.setAttribute("class", "device-routing-table");
            routingTableTitle.setAttribute("x", deviceData.width / 2);
            routingTableTitle.setAttribute("y", yPosition);
            routingTableTitle.setAttribute("text-anchor", "middle");
            routingTableTitle.setAttribute("font-size", "10");
            routingTableTitle.setAttribute("font-weight", "bold");
            routingTableTitle.setAttribute("fill", "#3498db");
            device.element.appendChild(routingTableTitle);
            
            yPosition += 15;
            
            // 显示每个路由条目
            device.properties.routingTable.forEach((route, index) => {
                if (route.network && route.subnetMask) {
                    const maskLength = getMaskLength(route.subnetMask);
                    const routeText = document.createElementNS("http://www.w3.org/2000/svg", "text");
                    routeText.textContent = `${route.network}/${maskLength} ${route.nextHop || "直接连接"}`;
                    routeText.setAttribute("class", "device-routing-table");
                    routeText.setAttribute("x", deviceData.width / 2);
                    routeText.setAttribute("y", yPosition);
                    routeText.setAttribute("text-anchor", "middle");
                    routeText.setAttribute("font-size", "9");
                    routeText.setAttribute("fill", "#555");
                    device.element.appendChild(routeText);
                    
                    yPosition += 15;
                }
            });
        }
    } else {
        // 其他设备类型的显示逻辑
        let ipAddress = "";
        let gatewayAddress = "";
        let dnsAddress = "";
        
        // 获取所有有IP属性的设备类型的IP地址和网关
        if (device.type === 'computer' || device.type === 'laptop' || device.type === 'server' || 
            device.type === 'dns-server' || device.type === 'web-server' || device.type === 'email-server') {
            
            // 对于笔记本，需要根据IP模式决定显示什么
            if (device.type === 'laptop' && device.properties.ipMode === 'dhcp') {
                // DHCP模式：只显示DHCP分配的信息
                // 检查是否有匹配的路由器并已分配IP
                const wirelessRouter = devices.find(router => 
                    router.type === 'wireless-router' && 
                    router.properties.ssid === device.properties.ssid
                );
                
                if (wirelessRouter) {
                    // 有匹配的路由器，显示DHCP分配的信息
                    ipAddress = device.properties.ip || "";
                    gatewayAddress = device.properties.gateway || "";
                    dnsAddress = device.properties.dns || "";
                } else {
                    // 没有匹配的路由器，不显示IP信息（等待DHCP分配）
                    ipAddress = "";
                    gatewayAddress = "";
                    dnsAddress = "";
                }
            } else {
                // 静态模式或其他设备类型：显示配置的IP信息
                ipAddress = device.properties.ip || "";
                gatewayAddress = device.properties.gateway || "";
                dnsAddress = device.properties.dns || "";
            }
        }
        
        // 重新创建IP和网关显示
        if (ipAddress) {
            // 已经在函数开始处检查过deviceData，直接使用
            let yPosition = deviceData.height + 30;
            
            const ipText = document.createElementNS("http://www.w3.org/2000/svg", "text");
            ipText.textContent = `IP: ${ipAddress}`;
            ipText.setAttribute("class", "device-ip");
            ipText.setAttribute("x", deviceData.width / 2);
            ipText.setAttribute("y", yPosition);
            ipText.setAttribute("text-anchor", "middle");
            ipText.setAttribute("font-size", "10");
            ipText.setAttribute("fill", "#7f8c8d");
            device.element.appendChild(ipText);
            
            // 显示网关（如果有）
            if (gatewayAddress && (device.type === 'computer' || device.type === 'laptop' || device.type === 'server' || 
                device.type === 'dns-server' || device.type === 'web-server' || device.type === 'email-server')) {
                yPosition += 15;
                const gatewayText = document.createElementNS("http://www.w3.org/2000/svg", "text");
                gatewayText.textContent = `网关: ${gatewayAddress}`;
                gatewayText.setAttribute("class", "device-gateway");
                gatewayText.setAttribute("x", deviceData.width / 2);
                gatewayText.setAttribute("y", yPosition);
                gatewayText.setAttribute("text-anchor", "middle");
                gatewayText.setAttribute("font-size", "10");
                gatewayText.setAttribute("fill", "#7f8c8d");
                device.element.appendChild(gatewayText);
            }
            
            // 显示DNS服务器（特别是笔记本的DHCP分配信息）
            if (dnsAddress && (device.type === 'computer' || device.type === 'laptop' || device.type === 'server' || 
                device.type === 'dns-server' || device.type === 'web-server' || device.type === 'email-server')) {
                yPosition += 15;
                const dnsText = document.createElementNS("http://www.w3.org/2000/svg", "text");
                dnsText.textContent = `DNS: ${dnsAddress}`;
                dnsText.setAttribute("class", "device-dns");
                dnsText.setAttribute("x", deviceData.width / 2);
                dnsText.setAttribute("y", yPosition);
                dnsText.setAttribute("text-anchor", "middle");
                dnsText.setAttribute("font-size", "10");
                dnsText.setAttribute("fill", "#7f8c8d");
                device.element.appendChild(dnsText);
            }
            
            // 对于邮件服务器，显示邮件域名
            if (device.type === 'email-server' && device.properties.domain) {
                yPosition += 15;
                const domainText = document.createElementNS("http://www.w3.org/2000/svg", "text");
                domainText.textContent = `域名: ${device.properties.domain}`;
                domainText.setAttribute("class", "device-domain");
                domainText.setAttribute("x", deviceData.width / 2);
                domainText.setAttribute("y", yPosition);
                domainText.setAttribute("text-anchor", "middle");
                domainText.setAttribute("font-size", "10");
                domainText.setAttribute("fill", "#7f8c8d");
                device.element.appendChild(domainText);
            }
        }
    }
}

// 添加接口
function addInterface(deviceId) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && (device.type === 'router' || device.type === 'wireless-router')) {
        // 在添加新接口之前，先保存当前界面上的接口配置
        const interfaces = device.properties.interfaces;
        for (let index = 0; index < interfaces.length; index++) {
            const ifaceName = document.getElementById(`iface-name-${device.id}-${index}`);
            const ifaceIp = document.getElementById(`iface-ip-${device.id}-${index}`);
            const ifaceSubnet = document.getElementById(`iface-subnet-${device.id}-${index}`);
            
            if (ifaceName && ifaceIp && ifaceSubnet) {
                interfaces[index].name = ifaceName.value;
                interfaces[index].ip = ifaceIp.value;
                interfaces[index].subnetMask = ifaceSubnet.value;
            }
        }
        
        // 创建新接口
        const interfaceCount = device.properties.interfaces.length;
        const newInterface = {
            name: `eth${interfaceCount}`,
            ip: `192.168.${interfaceCount + 1}.1`,
            subnetMask: "255.255.255.0"
        };
        
        // 添加到接口列表
        device.properties.interfaces.push(newInterface);
        
        // 记录日志
        log(`添加接口: ${device.name} - ${newInterface.name}`);
        
        // 重新选择设备以更新属性面板
        selectDevice(device);
    }
}

// 删除接口
function removeInterface(deviceId, index) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && (device.type === 'router' || device.type === 'wireless-router')) {
        // 在删除接口之前，先保存当前界面上的接口配置
        const interfaces = device.properties.interfaces;
        for (let i = 0; i < interfaces.length; i++) {
            const ifaceName = document.getElementById(`iface-name-${device.id}-${i}`);
            const ifaceIp = document.getElementById(`iface-ip-${device.id}-${i}`);
            const ifaceSubnet = document.getElementById(`iface-subnet-${device.id}-${i}`);
            
            if (ifaceName && ifaceIp && ifaceSubnet) {
                interfaces[i].name = ifaceName.value;
                interfaces[i].ip = ifaceIp.value;
                interfaces[i].subnetMask = ifaceSubnet.value;
            }
        }
        
        // 确保至少保留一个接口
        if (device.properties.interfaces.length > 1) {
            // 删除接口
            const removedInterface = device.properties.interfaces[index];
            device.properties.interfaces.splice(index, 1);
            
            // 记录日志
            log(`删除接口: ${device.name} - ${removedInterface.name}`);
            
            // 重新选择设备以更新属性面板
            selectDevice(device);
        } else {
            alert('路由器至少需要保留一个接口！');
        }
    }
}

// 添加静态路由
function addRoute(deviceId) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && (device.type === 'router' || device.type === 'wireless-router')) {
        // 在添加路由之前，先保存当前界面上的接口配置
        const interfaces = device.properties.interfaces;
        for (let index = 0; index < interfaces.length; index++) {
            const ifaceName = document.getElementById(`iface-name-${device.id}-${index}`);
            const ifaceIp = document.getElementById(`iface-ip-${device.id}-${index}`);
            const ifaceSubnet = document.getElementById(`iface-subnet-${device.id}-${index}`);
            
            if (ifaceName && ifaceIp && ifaceSubnet) {
                interfaces[index].name = ifaceName.value;
                interfaces[index].ip = ifaceIp.value;
                interfaces[index].subnetMask = ifaceSubnet.value;
            }
        }
        
        // 创建新路由表项
        const newRoute = {
            network: "0.0.0.0",
            subnetMask: "0.0.0.0",
            nextHop: ""
        };
        
        // 添加到路由表
        device.properties.routingTable.push(newRoute);
        
        // 记录日志
        log(`添加静态路由: ${device.name}`);
        
        // 更新设备显示
        updateDeviceIPDisplay(device);
        
        // 重新选择设备以更新属性面板
        selectDevice(device);
    }
}

// 删除静态路由
function removeRoute(deviceId, index) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && (device.type === 'router' || device.type === 'wireless-router')) {
        // 在删除路由之前，先保存当前界面上的接口配置和路由配置
        
        // 保存接口配置
        const interfaces = device.properties.interfaces;
        for (let i = 0; i < interfaces.length; i++) {
            const ifaceName = document.getElementById(`iface-name-${device.id}-${i}`);
            const ifaceIp = document.getElementById(`iface-ip-${device.id}-${i}`);
            const ifaceSubnet = document.getElementById(`iface-subnet-${device.id}-${i}`);
            
            if (ifaceName && ifaceIp && ifaceSubnet) {
                interfaces[i].name = ifaceName.value;
                interfaces[i].ip = ifaceIp.value;
                interfaces[i].subnetMask = ifaceSubnet.value;
            }
        }
        
        // 保存路由表配置
        const routes = device.properties.routingTable;
        for (let i = 0; i < routes.length; i++) {
            const routeNetwork = document.getElementById(`route-network-${device.id}-${i}`);
            const routeMask = document.getElementById(`route-mask-${device.id}-${i}`);
            const routeNextHop = document.getElementById(`route-next-hop-${device.id}-${i}`);
            
            if (routeNetwork && routeMask && routeNextHop) {
                routes[i].network = routeNetwork.value;
                routes[i].subnetMask = routeMask.value;
                routes[i].nextHop = routeNextHop.value;
            }
        }
        
        // 删除路由表项
        const removedRoute = device.properties.routingTable[index];
        device.properties.routingTable.splice(index, 1);
        
        // 记录日志
        log(`删除静态路由: ${device.name} - ${removedRoute.network}/${removedRoute.subnetMask}`);
        
        // 更新设备显示
        updateDeviceIPDisplay(device);
        
        // 重新选择设备以更新属性面板
        selectDevice(device);
    }
}

// 添加域名映射
function addDomain(deviceId) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && device.type === 'dns-server') {
        // 创建新域名映射
        const domainCount = device.properties.domains.length;
        const newDomain = {
            name: `example${domainCount + 1}.com`,
            ip: `192.168.1.${60 + domainCount}`,
            type: 'A' // 默认使用A记录类型
        };
        
        // 添加到域名列表
        device.properties.domains.push(newDomain);
        
        // 记录日志
        log(`添加域名映射: ${device.name} - ${newDomain.name} -> ${newDomain.ip} (${newDomain.type})`);
        
        // 重新选择设备以更新属性面板
        selectDevice(device);
    }
}

// 删除域名映射
function removeDomain(deviceId, index) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && device.type === 'dns-server') {
        // 确保至少保留一个域名映射
        if (device.properties.domains.length > 1) {
            // 删除域名映射
            const removedDomain = device.properties.domains[index];
            device.properties.domains.splice(index, 1);
            
            // 记录日志
            log(`删除域名映射: ${device.name} - ${removedDomain.name} -> ${removedDomain.ip} (${removedDomain.type || 'A'})`);
            
            // 重新选择设备以更新属性面板
            selectDevice(device);
        } else {
            alert('DNS服务器至少需要保留一个域名映射！');
        }
    }
}

// 添加Email用户
function addEmailUser(deviceId) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && device.type === 'email-server') {
        // 创建新用户
        const userCount = device.properties.users.length;
        const newUser = {
            username: `user${userCount + 1}`,
            password: `password${userCount + 1}`,
            emails: []
        };
        
        // 添加到用户列表
        device.properties.users.push(newUser);
        
        // 记录日志
        log(`添加邮件用户: ${device.name} - ${newUser.username}`);
        
        // 重新选择设备以更新属性面板
        selectDevice(device);
    }
}

// 删除Email用户
function removeEmailUser(deviceId, index) {
    // 仿真运行时不能修改设备属性
    if (isSimulating) {
        alert('仿真运行时不能修改设备属性，请先停止仿真');
        return;
    }
    
    const device = devices.find(d => d.id === deviceId);
    if (device && device.type === 'email-server') {
        // 确保至少保留一个用户
        if (device.properties.users.length > 1) {
            // 删除用户
            const removedUser = device.properties.users[index];
            device.properties.users.splice(index, 1);
            
            // 记录日志
            log(`删除邮件用户: ${device.name} - ${removedUser.username}`);
            
            // 重新选择设备以更新属性面板
            selectDevice(device);
        } else {
            alert('邮件服务器至少需要保留一个用户！');
        }
    }
}

// 保存连接属性
function saveConnectionProperties() {
    // 仿真运行时不能修改连接属性
    if (isSimulating) {
        alert('仿真运行时不能修改连接属性，请先停止仿真');
        return;
    }
    
    if (!selectedConnection) return;
    
    // 更新连接状态
    const status = document.getElementById('connection-status').value;
    selectedConnection.isActive = (status === 'active');
    
    // 更新连接样式
    if (selectedConnection.isActive) {
        selectedConnection.element.style.stroke = '#3498db';
    } else {
        selectedConnection.element.style.stroke = '#95a5a6';
    }
    
    log(`更新连接属性: ${selectedConnection.device1.name} ↔ ${selectedConnection.device2.name}`);
}

// 删除设备
function deleteDevice(deviceId) {
    // 仿真运行时不能删除设备
    if (isSimulating) {
        alert('仿真运行时不能删除设备，请先停止仿真');
        return;
    }
    
    if (confirm('确定要删除此设备吗？')) {
        const device = devices.find(d => d.id === deviceId);
        if (device) {
            // 如果是WiFi路由器，先记录其SSID，以便更新相关笔记本
            const wasWirelessRouter = device.type === 'wireless-router';
            const routerSSID = wasWirelessRouter ? device.properties.ssid : null;
            
            // 删除相关连接
            device.connections.forEach(connection => {
                deleteConnectionById(connection.id);
            });
            
            // 从设备列表中移除
            devices = devices.filter(d => d.id !== deviceId);
            
            // 删除SVG元素
            canvas.removeChild(device.element);
            
            // 如果是WiFi路由器被删除，更新所有使用DHCP模式的笔记本
            if (wasWirelessRouter && routerSSID) {
                updateLaptopsAfterRouterChange(routerSSID);
            }
            
            // 重置选择
            if (selectedDevice === device) {
                selectedDevice = null;
                document.getElementById('device-properties').innerHTML = `<h3>设备属性</h3><div class="properties-content"><p>选择设备查看和编辑属性</p></div>`;
            }
            
            log(`删除设备: ${device.name}`);
        }
    }
}

// 删除连接
function deleteConnection() {
    // 仿真运行时不能删除连接
    if (isSimulating) {
        alert('仿真运行时不能删除连接，请先停止仿真');
        return;
    }
    
    if (selectedConnection) {
        deleteConnectionById(selectedConnection.id);
    }
}

// 根据ID删除连接
function deleteConnectionById(connectionId) {
    // 仿真运行时不能删除连接
    if (isSimulating) {
        alert('仿真运行时不能删除连接，请先停止仿真');
        return;
    }
    
    const connection = connections.find(conn => conn.id === connectionId);
    if (connection) {
        // 从设备连接列表中移除
        connection.device1.connections = connection.device1.connections.filter(conn => conn.id !== connectionId);
        connection.device2.connections = connection.device2.connections.filter(conn => conn.id !== connectionId);
        
        // 从连接列表中移除
        connections = connections.filter(conn => conn.id !== connectionId);
        
        // 删除SVG元素
        canvas.removeChild(connection.element);
        canvas.removeChild(connection.dots[0]);
        canvas.removeChild(connection.dots[1]);
        
        // 重置选择
        if (selectedConnection === connection) {
            selectedConnection = null;
            document.getElementById('connection-properties').innerHTML = `<h3>连接属性</h3><div class="properties-content"><p>选择连接查看和编辑属性</p></div>`;
        }
        
        log(`删除连接: ${connection.device1.name} ↔ ${connection.device2.name}`);
    }
}

// 切换连接模式
function toggleConnectMode() {
    // 仿真运行时不能创建连接
    if (isSimulating) {
        alert('仿真运行时不能创建连接，请先停止仿真');
        return;
    }
    
    isConnecting = !isConnecting;
    const button = document.getElementById('connect-devices');
    
    if (isConnecting) {
        button.classList.add('active');
        button.style.backgroundColor = '#e74c3c';
        log("进入连接模式，请选择第一个设备");
    } else {
        button.classList.remove('active');
        button.style.backgroundColor = '#3498db';
        startDevice = null;
        log("退出连接模式");
    }
}

// 使设备可拖拽
function makeDraggable(element) {
    // 鼠标按下事件
    element.addEventListener('mousedown', function(e) {
        // 仿真运行时不能移动设备
        if (isSimulating) {
            alert('仿真运行时不能移动设备，请先停止仿真');
            return;
        }
        
        // 只有当没有其他拖拽操作时才开始
        if (!isDragging) {
            isDragging = true;
            draggedElement = this;
            
            const device = devices.find(d => d.element === this);
            if (device) {
                selectDevice(device);
            }
            
            // 记录初始位置
            const transform = this.getAttribute('transform');
            const match = transform.match(/translate\(([^,]+),\s*([^)]+)\)/);
            if (match) {
                elementX = parseFloat(match[1]);
                elementY = parseFloat(match[2]);
            } else {
                elementX = 0;
                elementY = 0;
            }
            
            // 计算鼠标相对于SVG画布的坐标（考虑滚动）
            const canvasContainer = document.querySelector('.canvas-container');
            const canvasRect = canvas.getBoundingClientRect();
            const scrollLeft = canvasContainer ? canvasContainer.scrollLeft : 0;
            const scrollTop = canvasContainer ? canvasContainer.scrollTop : 0;
            startX = e.clientX - canvasRect.left + scrollLeft;
            startY = e.clientY - canvasRect.top + scrollTop;
            
            // 添加拖拽类
            this.classList.add('dragging');
            
            e.preventDefault();
        }
    });
    
    // 触摸开始事件
    element.addEventListener('touchstart', function(e) {
        // 仿真运行时不能移动设备
        if (isSimulating) {
            alert('仿真运行时不能移动设备，请先停止仿真');
            return;
        }
        
        // 只有当没有其他拖拽操作时才开始
        if (!isDragging) {
            isDragging = true;
            draggedElement = this;
            
            const device = devices.find(d => d.element === this);
            if (device) {
                selectDevice(device);
            }
            
            // 记录初始位置
            const transform = this.getAttribute('transform');
            const match = transform.match(/translate\(([^,]+),\s*([^)]+)\)/);
            if (match) {
                elementX = parseFloat(match[1]);
                elementY = parseFloat(match[2]);
            } else {
                elementX = 0;
                elementY = 0;
            }
            
            // 计算触摸相对于SVG画布的坐标（考虑滚动）
            const canvasContainer = document.querySelector('.canvas-container');
            const canvasRect = canvas.getBoundingClientRect();
            const scrollLeft = canvasContainer ? canvasContainer.scrollLeft : 0;
            const scrollTop = canvasContainer ? canvasContainer.scrollTop : 0;
            const touch = e.touches[0];
            startX = touch.clientX - canvasRect.left + scrollLeft;
            startY = touch.clientY - canvasRect.top + scrollTop;
            
            // 添加拖拽类
            this.classList.add('dragging');
            
            e.preventDefault();
        }
    });
}

// 连接更新相关变量
let connectionUpdateTimeout = null;
let lastConnectionUpdateTime = 0;
const CONNECTION_UPDATE_INTERVAL = 16; // 约60fps

// 全局鼠标移动事件监听器
document.addEventListener('mousemove', function(e) {
    if (isDragging && draggedElement) {
        const device = devices.find(d => d.element === draggedElement);
        if (device) {
            // 计算新位置（考虑滚动）
            const canvasContainer = document.querySelector('.canvas-container');
            const canvasRect = canvas.getBoundingClientRect();
            const scrollLeft = canvasContainer ? canvasContainer.scrollLeft : 0;
            const scrollTop = canvasContainer ? canvasContainer.scrollTop : 0;
            const mouseX = e.clientX - canvasRect.left + scrollLeft;
            const mouseY = e.clientY - canvasRect.top + scrollTop;
            
            let newX = elementX + (mouseX - startX);
            let newY = elementY + (mouseY - startY);
            
            // 画布尺寸（来自#canvas-wrapper的CSS设置）
            const canvasWidth = 2000;
            const canvasHeight = 2000;
            
            // 限制设备在画布范围内
            newX = Math.max(0, Math.min(newX, canvasWidth - device.width));
            newY = Math.max(0, Math.min(newY, canvasHeight - device.height));
            
            // 1. 立即更新设备位置（同步）
            // 这是最重要的一步，确保设备位置实时更新
            draggedElement.setAttribute('transform', `translate(${newX}, ${newY})`);
            device.x = newX;
            device.y = newY;
            
            // 2. 立即更新相关连接，确保连接线与设备图标同时移动
            updateDeviceConnections(device);
            
            // 更新WiFi连接虚线
            updateWirelessConnections();
        }
    }
});

// 全局触摸移动事件监听器
document.addEventListener('touchmove', function(e) {
    if (isDragging && draggedElement) {
        const device = devices.find(d => d.element === draggedElement);
        if (device) {
            // 计算新位置（考虑滚动）
            const canvasContainer = document.querySelector('.canvas-container');
            const canvasRect = canvas.getBoundingClientRect();
            const scrollLeft = canvasContainer ? canvasContainer.scrollLeft : 0;
            const scrollTop = canvasContainer ? canvasContainer.scrollTop : 0;
            const touch = e.touches[0];
            const mouseX = touch.clientX - canvasRect.left + scrollLeft;
            const mouseY = touch.clientY - canvasRect.top + scrollTop;
            
            let newX = elementX + (mouseX - startX);
            let newY = elementY + (mouseY - startY);
            
            // 画布尺寸（来自#canvas-wrapper的CSS设置）
            const canvasWidth = 2000;
            const canvasHeight = 2000;
            
            // 限制设备在画布范围内
            newX = Math.max(0, Math.min(newX, canvasWidth - device.width));
            newY = Math.max(0, Math.min(newY, canvasHeight - device.height));
            
            // 1. 立即更新设备位置（同步）
            // 这是最重要的一步，确保设备位置实时更新
            draggedElement.setAttribute('transform', `translate(${newX}, ${newY})`);
            device.x = newX;
            device.y = newY;
            
            // 2. 立即更新相关连接，确保连接线与设备图标同时移动
            updateDeviceConnections(device);
            
            // 更新WiFi连接虚线
            updateWirelessConnections();
        }
    }
});

// 全局鼠标释放事件监听器
document.addEventListener('mouseup', function() {
    if (isDragging && draggedElement) {
        draggedElement.classList.remove('dragging');
        isDragging = false;
        draggedElement = null;
    }
});

// 全局触摸结束事件监听器
document.addEventListener('touchend', function() {
    if (isDragging && draggedElement) {
        draggedElement.classList.remove('dragging');
        isDragging = false;
        draggedElement = null;
    }
});

// 更新设备连接
function updateDeviceConnections(device) {
    device.connections.forEach(connection => {
        const isDevice1 = connection.device1 === device;
        const centerX = device.x + device.width / 2;
        const centerY = device.y + device.height / 2;
        
        if (isDevice1) {
            connection.x1 = centerX;
            connection.y1 = centerY;
            connection.dots[0].setAttribute('cx', centerX);
            connection.dots[0].setAttribute('cy', centerY);
        } else {
            connection.x2 = centerX;
            connection.y2 = centerY;
            connection.dots[1].setAttribute('cx', centerX);
            connection.dots[1].setAttribute('cy', centerY);
        }
        
        // 更新连接线
        connection.element.setAttribute('x1', connection.x1);
        connection.element.setAttribute('y1', connection.y1);
        connection.element.setAttribute('x2', connection.x2);
        connection.element.setAttribute('y2', connection.y2);
    });
}

// 运行仿真
function runSimulation() {
    if (isSimulating) return;
    
    // 处理无线SSID匹配和IP分配
    devices.forEach(device => {
        if (device.type === 'laptop') {
            // 查找匹配的无线路由器（SSID匹配）
            const wirelessRouter = devices.find(router => 
                router.type === 'wireless-router' && 
                router.properties.ssid === device.properties.ssid
            );
            
            if (wirelessRouter) {
                // SSID匹配成功，检查密码是否正确
                let passwordMatch = true;
                const security = wirelessRouter.properties.security ? wirelessRouter.properties.security.toLowerCase() : '';
                if (security && security !== 'open') {
                    // 如果路由器设置了安全模式（非开放），需要验证密码
                    passwordMatch = (device.properties.password === wirelessRouter.properties.password);
                }
                
                if (passwordMatch) {
                    // SSID和密码都匹配成功
                    if (device.properties.ipMode === 'dhcp') {
                        // DHCP模式：从路由器获取IP地址、网关和DNS服务器
                        const routerInterface = wirelessRouter.properties.interfaces.find(iface => iface.name === 'wlan0') || wirelessRouter.properties.interfaces[0];
                        if (routerInterface) {
                            // 从路由器所在网段分配一个IP
                            const routerIP = routerInterface.ip;
                            const subnet = routerIP.substring(0, routerIP.lastIndexOf('.'));
                            // 简单分配一个IP，实际应该检查是否已分配
                            device.properties.ip = `${subnet}.${Math.floor(Math.random() * 254) + 1}`;
                            device.properties.subnetMask = routerInterface.subnetMask;
                            device.properties.gateway = routerIP;
                            // DNS服务器地址来自无线路由器的DHCP设置
                            device.properties.dns = wirelessRouter.properties.dhcpDns || routerIP;
                            log(`笔记本 ${device.name} 通过DHCP从路由器 ${wirelessRouter.name} 获取IP: ${device.properties.ip}, 网关: ${device.properties.gateway}, DNS: ${device.properties.dns}`);
                        }
                    }
                    // 静态模式：使用已配置的IP
                } else {
                    // 密码不匹配
                    log(`笔记本 ${device.name} 无法连接到WiFi路由器 ${wirelessRouter.name}：密码错误`);
                }
            } else {
                // SSID匹配失败
                log(`笔记本 ${device.name} 无法找到SSID: ${device.properties.ssid}`);
            }
        }
    });
    
    isSimulating = true;
    simulationStartTime = Date.now();
    packetCount = 0;
    
    // 更新状态
    safeSetTextContent('status-text', '运行中');
    const statusTextElement = document.getElementById('status-text');
    if (statusTextElement) {
        statusTextElement.style.color = '#27ae60';
    }
    
    // 更新设备IP和网关显示
    devices.forEach(device => {
        updateDeviceIPDisplay(device);
    });
    
    // 更新WiFi连接虚线
    updateWirelessConnections();
    
    // 启动仿真循环
    simulationInterval = setInterval(() => {
        updateSimulation();
    }, 1000);
    
    // 更新实验工具状态
    updateExperimentToolsStatus();
    
    log("开始网络仿真");
}

// 停止仿真
function stopSimulation() {
    if (!isSimulating) return;
    
    isSimulating = false;
    
    // 清除仿真循环
    clearInterval(simulationInterval);
    simulationInterval = null;
    
    // 更新状态
    safeSetTextContent('status-text', '未运行');
    const statusTextElement = document.getElementById('status-text');
    if (statusTextElement) {
        statusTextElement.style.color = '#7f8c8d';
    }
    
    // 更新按钮文本
    safeSetTextContent('toggle-simulation', '运行仿真');
    
    // 更新实验工具状态
    updateExperimentToolsStatus();
    
    log("停止网络仿真");
}

// 更新实验工具状态
function updateExperimentToolsStatus() {
    const toolIds = ['ping-tool', 'traceroute-tool', 'nslookup-tool', 'web-browser', 'email-tool', 'network-command-tool'];
    toolIds.forEach(toolId => {
        const tool = document.getElementById(toolId);
        if (tool) {
            if (isSimulating) {
                tool.disabled = false;
                tool.style.opacity = '1';
                tool.style.cursor = 'pointer';
            } else {
                tool.disabled = true;
                tool.style.opacity = '0.5';
                tool.style.cursor = 'not-allowed';
            }
        }
    });
}

// 删除仿真状态中的数据包显示
document.addEventListener('DOMContentLoaded', function() {
    const packetCountDiv = document.querySelector('#simulation-status .status-item:nth-child(3)');
    if (packetCountDiv) {
        packetCountDiv.remove();
    }
});

// 切换仿真状态
function toggleSimulation() {
    if (isSimulating) {
        stopSimulation();
    } else {
        runSimulation();
        // 更新按钮文本
        safeSetTextContent('toggle-simulation', '停止仿真');
    }
}

// 更新仿真
function updateSimulation() {
    if (!isSimulating) return;
    
    // 更新运行时间
    const elapsed = Date.now() - simulationStartTime;
    const hours = Math.floor(elapsed / (1000 * 60 * 60));
    const minutes = Math.floor((elapsed % (1000 * 60 * 60)) / (1000 * 60));
    const seconds = Math.floor((elapsed % (1000 * 60)) / 1000);
    const timeString = `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    safeSetTextContent('run-time', timeString);
    
    // 不自动模拟数据包传输，只在ping时显示数据包动画
    

}



// 创建沿路径传输的数据包
function createPacketAlongPath(path, sourceDevice, targetDevice) {
    // 如果路径只有两个设备，直接在连接上创建数据包
    if (path.length === 2) {
        const connection = getConnectionBetweenDevices(path[0], path[1]);
        if (connection) {
            animatePacketOnConnection(connection, sourceDevice, targetDevice);
        }
        return;
    }
    
    // 如果路径有多个设备，在每段连接上依次创建数据包
    for (let i = 0; i < path.length - 1; i++) {
        const currentDevice = path[i];
        const nextDevice = path[i + 1];
        
        const connection = getConnectionBetweenDevices(currentDevice, nextDevice);
        
        if (connection) {
            // 延迟创建下一段的数据包，模拟真实的转发过程
            setTimeout(() => {
                animatePacketOnConnection(connection, sourceDevice, targetDevice);
            }, i * 1000);
        }
    }
}

// 获取两个设备之间的连接（包括物理连接和无线连接）
function getConnectionBetweenDevices(device1, device2) {
    // 首先查找物理连接
    const physicalConnection = connections.find(conn => 
        (conn.device1 === device1 && conn.device2 === device2) || 
        (conn.device1 === device2 && conn.device2 === device1)
    );
    
    if (physicalConnection) {
        // 确保连接的端点顺序与路径中的设备顺序一致
        if (physicalConnection.device1 === device1 && physicalConnection.device2 === device2) {
            return physicalConnection;
        } else {
            // 连接的端点顺序与路径中的设备顺序相反，交换端点
            return {
                id: physicalConnection.id,
                device1: device1,
                device2: device2,
                x1: physicalConnection.x2,
                y1: physicalConnection.y2,
                x2: physicalConnection.x1,
                y2: physicalConnection.y1,
                element: physicalConnection.element,
                dots: physicalConnection.dots,
                isActive: physicalConnection.isActive
            };
        }
    }
    
    // 检查是否是无线连接
    if ((device1.type === 'laptop' && device2.type === 'wireless-router' && 
        device1.properties.ssid === device2.properties.ssid) ||
        (device1.type === 'wireless-router' && device2.type === 'laptop' && 
         device1.properties.ssid === device2.properties.ssid)) {
        
        // 计算设备中心点坐标，确保端点顺序与路径中的设备顺序一致
        const sourceCenterX = device1.x + device1.width / 2;
        const sourceCenterY = device1.y + device1.height / 2;
        const targetCenterX = device2.x + device2.width / 2;
        const targetCenterY = device2.y + device2.height / 2;
        
        // 创建虚拟连接对象
        return {
            x1: sourceCenterX,
            y1: sourceCenterY,
            x2: targetCenterX,
            y2: targetCenterY
        };
    }
    
    return null;
}

// 在指定连接上创建并动画化数据包
function animatePacketOnConnection(connection, sourceDevice, targetDevice) {
    // 计算路径
    const startX = connection.x1;
    const startY = connection.y1;
    const endX = connection.x2;
    const endY = connection.y2;
    
    // 创建数据包SVG元素
    const packet = document.createElementNS("http://www.w3.org/2000/svg", "circle");
    packet.setAttribute("class", "packet");
    packet.setAttribute("r", "3");
    packet.setAttribute("fill", "#e74c3c");
    packet.setAttribute("stroke", "white");
    packet.setAttribute("stroke-width", "1");
    packet.setAttribute("cx", startX);
    packet.setAttribute("cy", startY);
    canvas.appendChild(packet);
    
    // 动画参数
    const duration = 2000; // 2秒
    const steps = 50;
    const stepTime = duration / steps;
    
    // 计算每步移动距离
    const dx = (endX - startX) / steps;
    const dy = (endY - startY) / steps;
    
    // 动画循环
    let step = 0;
    const animatePacket = setInterval(() => {
        step++;
        const x = startX + dx * step;
        const y = startY + dy * step;
        packet.setAttribute("cx", x);
        packet.setAttribute("cy", y);
        
        if (step >= steps) {
            clearInterval(animatePacket);
            canvas.removeChild(packet);
        }
    }, stepTime);
}

// 新建拓扑
function newTopology() {
    // 仿真运行时不能新建拓扑
    if (isSimulating) {
        alert('仿真运行时不能新建拓扑，请先停止仿真');
        return;
    }
    
    if (confirm('确定要新建拓扑吗？当前拓扑将丢失。')) {
        // 停止仿真
        stopSimulation();
        
        // 删除所有连接
        connections.forEach(connection => {
            canvas.removeChild(connection.element);
            canvas.removeChild(connection.dots[0]);
            canvas.removeChild(connection.dots[1]);
        });
        connections = [];
        
        // 删除所有设备
        devices.forEach(device => {
            canvas.removeChild(device.element);
        });
        devices = [];
        
        // 重置选择
        selectedDevice = null;
        selectedConnection = null;
        
        // 重置属性面板
        document.getElementById('device-properties').innerHTML = `<h3>设备属性</h3><div class="properties-content"><p>选择设备查看和编辑属性</p></div>`;
        document.getElementById('connection-properties').innerHTML = `<h3>连接属性</h3><div class="properties-content"><p>选择连接查看和编辑属性</p></div>`;
        
        // 重置状态
        safeSetTextContent('status-text', '未运行');
        const statusTextElement = document.getElementById('status-text');
        if (statusTextElement) {
            statusTextElement.style.color = '#7f8c8d';
        }
        safeSetTextContent('run-time', '00:00:00');
        safeSetTextContent('packet-count', '0');
        
        log("新建拓扑");
        return true; // 用户点击了确定
    }
    return false; // 用户点击了取消
}

// 保存拓扑
function saveTopology() {
    // 仿真运行时不能保存拓扑
    if (isSimulating) {
        alert('仿真运行时不能保存拓扑，请先停止仿真');
        return;
    }
    
    const topology = {
        devices: devices.map(device => ({
            id: device.id,
            type: device.type,
            name: device.name,
            x: device.x,
            y: device.y,
            width: device.width,
            height: device.height,
            properties: JSON.parse(JSON.stringify(device.properties)) // 深拷贝确保只保存可序列化内容
        })),
        connections: connections.map(connection => ({
            id: connection.id,
            device1Id: connection.device1.id,
            device2Id: connection.device2.id,
            isActive: connection.isActive
        }))
    };
    
    // 转换为JSON字符串
    const json = JSON.stringify(topology, null, 2);
    
    // 创建下载链接
    const blob = new Blob([json], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `MixNET-topology-${Date.now()}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    log("拓扑保存成功");
}

// 打开拓扑
function openTopology() {
    // 仿真运行时不能打开拓扑
    if (isSimulating) {
        alert('仿真运行时不能打开拓扑，请先停止仿真');
        return;
    }
    
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = '.json';
    input.onchange = function(e) {
        const file = e.target.files[0];
        if (file) {
            // 先询问用户是否要覆盖原有拓扑
            if (!confirm('确定要打开新拓扑吗？当前拓扑将丢失。')) {
                return;
            }
            
            const reader = new FileReader();
            reader.onload = function(event) {
                try {
                    // 基本BOM头处理
                    let content = event.target.result;
                    if (content.charCodeAt(0) === 0xFEFF) {
                        content = content.slice(1);
                    }
                    
                    const topology = JSON.parse(content);
                    
                    // 停止仿真
                    stopSimulation();
                    
                    // 删除所有连接
                    connections.forEach(connection => {
                        canvas.removeChild(connection.element);
                        canvas.removeChild(connection.dots[0]);
                        canvas.removeChild(connection.dots[1]);
                    });
                    connections = [];
                    
                    // 删除所有设备
                    devices.forEach(device => {
                        canvas.removeChild(device.element);
                    });
                    devices = [];
                    
                    // 重置选择
                    selectedDevice = null;
                    selectedConnection = null;
                    
                    // 重置属性面板
                    document.getElementById('device-properties').innerHTML = `<h3>设备属性</h3><div class="properties-content"><p>选择设备查看和编辑属性</p></div>`;
                    document.getElementById('connection-properties').innerHTML = `<h3>连接属性</h3><div class="properties-content"><p>选择连接查看和编辑属性</p></div>`;
                    
                    // 重置状态
                    safeSetTextContent('status-text', '未运行');
                    const statusTextElement = document.getElementById('status-text');
                    if (statusTextElement) {
                        statusTextElement.style.color = '#7f8c8d';
                    }
                    safeSetTextContent('run-time', '00:00:00');
                    
                    log("新建拓扑");
                    
                    // 加载设备
                    const deviceMap = {};
                    topology.devices.forEach((deviceData, index) => {
                        log(`加载设备 ${index + 1}/${topology.devices.length}: ${deviceData.name} (${deviceData.type})`);
                        
                        // 验证设备类型是否存在
                        if (!DEVICE_TYPES[deviceData.type]) {
                            alert(`拓扑文件包含未知设备类型: ${deviceData.type}`);
                            log(`加载设备失败: 未知设备类型 ${deviceData.type}`);
                            return;
                        }
                        
                        const device = addDevice(
                            deviceData.type,
                            deviceData.x + deviceData.width / 2,
                            deviceData.y + deviceData.height / 2
                        );
                        
                        // 检查设备是否成功创建
                        if (device) {
                            device.id = deviceData.id; // 使用拓扑文件中的原始ID
                            device.element.id = deviceData.id; // 更新DOM元素ID
                            device.name = deviceData.name;
                            // 使用深拷贝确保设备属性的完整性
                            device.properties = JSON.parse(JSON.stringify(deviceData.properties));
                            // 确保设备有 connections 数组
                            if (!device.connections) {
                                device.connections = [];
                            }
                            // 确保路由器和无线路由器有 interfaces 属性
                            if ((device.type === 'router' || device.type === 'wireless-router') && !device.properties.interfaces) {
                                device.properties.interfaces = [];
                            }
                            // 确保交换机有 ports 属性
                            if (device.type === 'switch' && !device.properties.ports) {
                                device.properties.ports = 8;
                            }
                            deviceMap[deviceData.id] = device;
                            
                            // 更新设备标签
                            const label = device.element.querySelector('.device-label');
                            if (label) {
                                label.textContent = device.name;
                            }
                            
                            // 更新设备IP和网关显示
                            updateDeviceIPDisplay(device);
                        } else {
                            log(`加载设备失败: ${deviceData.name} (${deviceData.type})`);
                        }
                    });
                    
                    // 加载连接
                    topology.connections.forEach(connectionData => {
                        const device1 = deviceMap[connectionData.device1Id];
                        const device2 = deviceMap[connectionData.device2Id];
                        if (device1 && device2) {
                            const connection = connectDevices(device1, device2, false);
                            if (connection) {
                                connection.isActive = connectionData.isActive;
                                if (!connection.isActive) {
                                    connection.element.style.stroke = '#95a5a6';
                                }
                            }
                        }
                    });
                    
                    // 更新WiFi连接虚线
                    updateWirelessConnections();
                    
                    log("拓扑加载成功");
                    
                    // 拓扑加载完成后，选择第一个设备来显示其属性
                    if (devices.length > 0) {
                        selectDevice(devices[0]);
                    }
                } catch (error) {
                    alert('拓扑文件格式错误');
                    log(`加载拓扑失败: ${error.message}`);
                    console.error('拓扑文件解析错误:', error);
                }
            };
            reader.readAsText(file);
        }
    };
    input.click();
}

// 加载拓扑
function loadTopology(topology) {
    // 停止仿真
    stopSimulation();
    
    // 删除所有连接
    connections.forEach(connection => {
        canvas.removeChild(connection.element);
        canvas.removeChild(connection.dots[0]);
        canvas.removeChild(connection.dots[1]);
    });
    connections = [];
    
    // 删除所有设备
    devices.forEach(device => {
        canvas.removeChild(device.element);
    });
    devices = [];
    
    // 重置选择
    selectedDevice = null;
    selectedConnection = null;
    
    // 重置属性面板
    document.getElementById('device-properties').innerHTML = `<h3>设备属性</h3><div class="properties-content"><p>选择设备查看和编辑属性</p></div>`;
    document.getElementById('connection-properties').innerHTML = `<h3>连接属性</h3><div class="properties-content"><p>选择连接查看和编辑属性</p></div>`;
    
    // 重置状态
    safeSetTextContent('status-text', '未运行');
    const statusTextElement = document.getElementById('status-text');
    if (statusTextElement) {
        statusTextElement.style.color = '#7f8c8d';
    }
    safeSetTextContent('run-time', '00:00:00');
    
    log("新建拓扑");
    
    // 加载设备
    const deviceMap = {};
    try {
        topology.devices.forEach((deviceData, index) => {
            log(`加载设备 ${index + 1}/${topology.devices.length}: ${deviceData.name} (${deviceData.type})`);
            
            // 验证设备类型是否存在
            if (!DEVICE_TYPES[deviceData.type]) {
                alert(`拓扑文件包含未知设备类型: ${deviceData.type}`);
                log(`加载设备失败: 未知设备类型 ${deviceData.type}`);
                return;
            }
            
            const device = addDevice(
                deviceData.type,
                deviceData.x + deviceData.width / 2,
                deviceData.y + deviceData.height / 2
            );
            
            // 检查设备是否成功创建
            if (device) {
                device.id = deviceData.id; // 使用拓扑文件中的原始ID
                device.element.id = deviceData.id; // 更新DOM元素ID
                device.name = deviceData.name;
                // 使用深拷贝确保设备属性的完整性
                device.properties = JSON.parse(JSON.stringify(deviceData.properties));
                // 确保设备有 connections 数组
                if (!device.connections) {
                    device.connections = [];
                }
                // 确保路由器和无线路由器有 interfaces 属性
                if ((device.type === 'router' || device.type === 'wireless-router') && !device.properties.interfaces) {
                    device.properties.interfaces = [];
                }
                // 确保交换机有 ports 属性
                if (device.type === 'switch' && !device.properties.ports) {
                    device.properties.ports = 8;
                }
                deviceMap[deviceData.id] = device;
                
                // 更新设备标签
                const label = device.element.querySelector('.device-label');
                if (label) {
                    label.textContent = device.name;
                }
                
                // 更新设备IP和网关显示
                updateDeviceIPDisplay(device);
            } else {
                log(`加载设备失败: ${deviceData.name} (${deviceData.type})`);
            }
        });
        
        // 加载连接
        topology.connections.forEach(connectionData => {
            const device1 = deviceMap[connectionData.device1Id];
            const device2 = deviceMap[connectionData.device2Id];
            if (device1 && device2) {
                const connection = connectDevices(device1, device2, false);
                if (connection) {
                    connection.isActive = connectionData.isActive;
                    if (!connection.isActive) {
                        connection.element.style.stroke = '#95a5a6';
                    }
                }
            }
        });
        
        // 更新WiFi连接虚线
        updateWirelessConnections();
        
        log("拓扑加载成功");
        
        // 拓扑加载完成后，选择第一个设备来显示其属性
        if (devices.length > 0) {
            selectDevice(devices[0]);
        }
        
        // 将画布移动到左上角
        const canvasContainer = document.querySelector('.canvas-container');
        if (canvasContainer) {
            // 使用setTimeout确保DOM更新完成后再滚动
            setTimeout(() => {
                canvasContainer.scrollLeft = 0;
                canvasContainer.scrollTop = 0;
            }, 0);
        }
    } catch (error) {
        log(`加载拓扑失败: ${error.message}`);
        log(`错误堆栈: ${error.stack}`);
        console.error('拓扑加载错误:', error);
    }
}

// 预设实验数据（内置）
const presetExperiments = {
    'basic-network': {
        "devices": [
            {
                "id": "device_1768309558664_524",
                "type": "computer",
                "name": "PC1",
                "x": 204,
                "y": 137,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.101",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768309558667_383",
                "type": "computer",
                "name": "PC2",
                "x": 522,
                "y": 127,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.102",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768309558668_655",
                "type": "switch",
                "name": "交换机1",
                "x": 370,
                "y": 284,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            }
        ],
        "connections": [
            {
                "id": "connection_11",
                "device1Id": "device_1768309558664_524",
                "device2Id": "device_1768309558668_655",
                "isActive": true
            },
            {
                "id": "connection_12",
                "device1Id": "device_1768309558667_383",
                "device2Id": "device_1768309558668_655",
                "isActive": true
            }
        ]
    },
    'routing': {
        "devices": [
            {
                "id": "device_1768310143117_830",
                "type": "computer",
                "name": "PC1",
                "x": 174,
                "y": 222,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768310143119_33",
                "type": "computer",
                "name": "PC2",
                "x": 679,
                "y": 218,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.3.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.3.1",
                    "dns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768310143119_304",
                "type": "router",
                "name": "路由器A",
                "x": 332,
                "y": 283,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.1.1",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth1",
                            "ip": "192.168.2.1",
                            "subnetMask": "255.255.255.0"
                        }
                    ],
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.2.2"
                        }
                    ]
                }
            },
            {
                "id": "device_1768310143121_561",
                "type": "router",
                "name": "路由器B",
                "x": 512,
                "y": 288,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.2.2",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth1",
                            "ip": "192.168.3.1",
                            "subnetMask": "255.255.255.0"
                        }
                    ],
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.2.1"
                        }
                    ]
                }
            },
            {
                "id": "device_1768310143122_313",
                "type": "switch",
                "name": "交换机1",
                "x": 334,
                "y": 137,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            },
            {
                "id": "device_1768310143123_682",
                "type": "switch",
                "name": "交换机2",
                "x": 516,
                "y": 135,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            }
        ],
        "connections": [
            {
                "id": "connection_1",
                "device1Id": "device_1768310143117_830",
                "device2Id": "device_1768310143122_313",
                "isActive": true
            },
            {
                "id": "connection_2",
                "device1Id": "device_1768310143122_313",
                "device2Id": "device_1768310143119_304",
                "isActive": true
            },
            {
                "id": "connection_3",
                "device1Id": "device_1768310143119_304",
                "device2Id": "device_1768310143121_561",
                "isActive": true
            },
            {
                "id": "connection_4",
                "device1Id": "device_1768310143121_561",
                "device2Id": "device_1768310143123_682",
                "isActive": true
            },
            {
                "id": "connection_5",
                "device1Id": "device_1768310143123_682",
                "device2Id": "device_1768310143119_33",
                "isActive": true
            }
        ]
    },
    'http-server': {
        "devices": [
            {
                "id": "device_1768310308343_820",
                "type": "computer",
                "name": "客户端",
                "x": 171,
                "y": 170,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768310308345_298",
                "type": "web-server",
                "name": "Web服务器",
                "x": 470,
                "y": 170,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.50",
                    "subnetMask": "255.255.255.0",
                    "website": {
                        "title": "欢迎页面",
                        "content": "<h1>这是一个简单的Web服务器页面</h1>\n<h2>你可以体会Web服务器的作用</h2>"
                    },
                    "gateway": "192.168.1.1",
                    "dns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768310308347_935",
                "type": "switch",
                "name": "交换机",
                "x": 315,
                "y": 175,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            }
        ],
        "connections": [
            {
                "id": "connection_6",
                "device1Id": "device_1768310308343_820",
                "device2Id": "device_1768310308347_935",
                "isActive": true
            },
            {
                "id": "connection_7",
                "device1Id": "device_1768310308345_298",
                "device2Id": "device_1768310308347_935",
                "isActive": true
            }
        ]
    },
    'dns-server': {
        "devices": [
            {
                "id": "device_1768530514669_965",
                "type": "computer",
                "name": "客户端1",
                "x": 170,
                "y": 120,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768530514671_204",
                "type": "computer",
                "name": "客户端2",
                "x": 172,
                "y": 277,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.101",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768530514672_57",
                "type": "dns-server",
                "name": "DNS服务器",
                "x": 471,
                "y": 198,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.50",
                    "subnetMask": "255.255.255.0",
                    "domains": [
                        {
                            "name": "mixly.cn",
                            "ip": "192.168.1.100",
                            "type": "A"
                        },
                        {
                            "name": "mixio.cn",
                            "ip": "192.168.1.101",
                            "type": "A"
                        },
                        {
                            "name": "mixly.cn",
                            "ip": "192.168.1.100",
                            "type": "MX"
                        }
                    ],
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768530514674_581",
                "type": "switch",
                "name": "交换机",
                "x": 315,
                "y": 201,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            }
        ],
        "connections": [
            {
                "id": "connection_1",
                "device1Id": "device_1768530514669_965",
                "device2Id": "device_1768530514674_581",
                "isActive": true
            },
            {
                "id": "connection_2",
                "device1Id": "device_1768530514671_204",
                "device2Id": "device_1768530514674_581",
                "isActive": true
            },
            {
                "id": "connection_3",
                "device1Id": "device_1768530514672_57",
                "device2Id": "device_1768530514674_581",
                "isActive": true
            }
        ]
    },
    'email-server': {
        "devices": [
            {
                "id": "device_1768308882697_551",
                "type": "computer",
                "name": "客户端1",
                "x": 137,
                "y": 63,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768308882709_418",
                "type": "computer",
                "name": "客户端2",
                "x": 164,
                "y": 280,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.101",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768308882710_470",
                "type": "dns-server",
                "name": "DNS服务器",
                "x": 354,
                "y": 63,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.50",
                    "subnetMask": "255.255.255.0",
                    "domains": [
                        {
                            "name": "163.com",
                            "ip": "192.168.1.70",
                            "type": "MX"
                        },
                        {
                            "name": "263.com",
                            "ip": "192.168.1.60",
                            "type": "MX"
                        }
                    ],
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768308882716_188",
                "type": "email-server",
                "name": "邮件服务器1",
                "x": 581,
                "y": 65,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.60",
                    "subnetMask": "255.255.255.0",
                    "domain": "263.com",
                    "smtpPort": 25,
                    "pop3Port": 110,
                    "imapPort": 143,
                    "users": [
                        {
                            "username": "263",
                            "password": "263",
                            "emails": []
                        }
                    ],
                    "emails": [],
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768308882721_574",
                "type": "email-server",
                "name": "邮件服务器2",
                "x": 581,
                "y": 303,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.70",
                    "subnetMask": "255.255.255.0",
                    "domain": "163.com",
                    "smtpPort": 25,
                    "pop3Port": 110,
                    "imapPort": 143,
                    "users": [
                        {
                            "username": "163",
                            "password": "163",
                            "emails": []
                        },
                        {
                            "username": "164",
                            "password": "164",
                            "emails": []
                        }
                    ],
                    "emails": [],
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768308882723_612",
                "type": "switch",
                "name": "交换机",
                "x": 365,
                "y": 225,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            }
        ],
        "connections": [
            {
                "id": "connection_1",
                "device1Id": "device_1768308882697_551",
                "device2Id": "device_1768308882723_612",
                "isActive": true
            },
            {
                "id": "connection_2",
                "device1Id": "device_1768308882709_418",
                "device2Id": "device_1768308882723_612",
                "isActive": true
            },
            {
                "id": "connection_3",
                "device1Id": "device_1768308882710_470",
                "device2Id": "device_1768308882723_612",
                "isActive": true
            },
            {
                "id": "connection_4",
                "device1Id": "device_1768308882716_188",
                "device2Id": "device_1768308882723_612",
                "isActive": true
            },
            {
                "id": "connection_5",
                "device1Id": "device_1768308882721_574",
                "device2Id": "device_1768308882723_612",
                "isActive": true
            }
        ]
    },
    'wifi-network': {
        "devices": [
            {
                "id": "device_1768526662333_805",
                "type": "laptop",
                "name": "笔记本1",
                "x": 127,
                "y": 69,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.147",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "8.8.8.8",
                    "ipMode": "dhcp",
                    "ssid": "WiFiNetwork",
                    "wirelessEnabled": true,
                    "password": "12345678"
                }
            },
            {
                "id": "device_1768526662336_320",
                "type": "laptop",
                "name": "笔记本2",
                "x": 126,
                "y": 250,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.101",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "8.8.8.8",
                    "ipMode": "static",
                    "ssid": "WiFiNetwork",
                    "wirelessEnabled": true,
                    "password": "12345678"
                }
            },
            {
                "id": "device_1768526662337_374",
                "type": "switch",
                "name": "交换机",
                "x": 565,
                "y": 175,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            },
            {
                "id": "device_1768526662338_744",
                "type": "computer",
                "name": "台式机",
                "x": 770,
                "y": 169,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "10.0.0.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "10.0.0.1",
                    "dns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768526662339_815",
                "type": "wireless-router",
                "name": "WiFi路由器",
                "x": 366,
                "y": 159,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "10.0.0.1",
                            "subnetMask": "255.255.255.0",
                            "role": "wan"
                        },
                        {
                            "name": "wlan0",
                            "ip": "192.168.1.1",
                            "subnetMask": "255.255.255.0",
                            "role": "lan"
                        }
                    ],
                    "ssid": "WiFiNetwork",
                    "security": "WPA2-PSK",
                    "password": "12345678",
                    "natEnabled": true,
                    "routingTable": [],
                    "dhcpDns": "8.8.8.8"
                }
            }
        ],
        "connections": [
            {
                "id": "connection_1",
                "device1Id": "device_1768526662337_374",
                "device2Id": "device_1768526662338_744",
                "isActive": true
            },
            {
                "id": "connection_2",
                "device1Id": "device_1768526662339_815",
                "device2Id": "device_1768526662337_374",
                "isActive": true
            }
        ]
    },
    'integrated-network': {
        "devices": [
            {
                "id": "device_1768527901890_891",
                "type": "computer",
                "name": "客户端1",
                "x": 194,
                "y": 113,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768527901892_861",
                "type": "computer",
                "name": "客户端2",
                "x": 198,
                "y": 288,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.101",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768527901893_879",
                "type": "dns-server",
                "name": "DNS服务器",
                "x": 354,
                "y": 63,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.50",
                    "subnetMask": "255.255.255.0",
                    "domains": [
                        {
                            "name": "163.com",
                            "ip": "192.168.3.70",
                            "type": "MX"
                        },
                        {
                            "name": "263.com",
                            "ip": "192.168.3.60",
                            "type": "MX"
                        }
                    ],
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768527901895_959",
                "type": "email-server",
                "name": "邮件服务器1",
                "x": 535,
                "y": 63,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.3.60",
                    "subnetMask": "255.255.255.0",
                    "domain": "263.com",
                    "smtpPort": 25,
                    "pop3Port": 110,
                    "imapPort": 143,
                    "users": [
                        {
                            "username": "263",
                            "password": "263",
                            "emails": [
                                {
                                    "id": 1768308882727,
                                    "from": "bob@test.com",
                                    "to": "alice@example.com",
                                    "subject": "测试邮件",
                                    "content": "这是一封测试邮件",
                                    "timestamp": "2026/1/13 20:54:42",
                                    "read": true
                                },
                                {
                                    "id": 1768309095172,
                                    "from": "163@163.com",
                                    "to": "263@263.com",
                                    "subject": "163@163.com",
                                    "content": "163@163.com",
                                    "timestamp": "2026/1/13 20:58:15",
                                    "read": true
                                },
                                {
                                    "id": 1768309416274,
                                    "from": "163@163.com",
                                    "to": "263@263.com",
                                    "subject": "new ",
                                    "content": "ew ew ew ew ",
                                    "timestamp": "2026/1/13 21:03:36",
                                    "read": true
                                },
                                {
                                    "id": 1768311671804,
                                    "from": "163@163.com",
                                    "to": "263@263.com",
                                    "subject": "1",
                                    "content": "111",
                                    "timestamp": "2026/1/13 21:41:11",
                                    "read": false
                                },
                                {
                                    "id": 1768311720992,
                                    "from": "163@163.com",
                                    "to": "263@263.com",
                                    "subject": "11",
                                    "content": "111",
                                    "timestamp": "2026/1/13 21:42:00",
                                    "read": false
                                }
                            ]
                        }
                    ],
                    "emails": [],
                    "gateway": "192.168.3.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768527901898_555",
                "type": "email-server",
                "name": "邮件服务器2",
                "x": 679,
                "y": 104,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.3.70",
                    "subnetMask": "255.255.255.0",
                    "domain": "163.com",
                    "smtpPort": 25,
                    "pop3Port": 110,
                    "imapPort": 143,
                    "users": [
                        {
                            "username": "163",
                            "password": "163",
                            "emails": []
                        },
                        {
                            "username": "164",
                            "password": "164",
                            "emails": []
                        }
                    ],
                    "emails": [],
                    "gateway": "192.168.3.1",
                    "dns": "192.168.1.50"
                }
            },
            {
                "id": "device_1768527901899_259",
                "type": "switch",
                "name": "交换机",
                "x": 365,
                "y": 225,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            },
            {
                "id": "device_1768527901900_784",
                "type": "router",
                "name": "路由器_7",
                "x": 362,
                "y": 347,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.1.1",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth1",
                            "ip": "192.168.2.1",
                            "subnetMask": "255.255.255.0"
                        }
                    ],
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.2.2"
                        }
                    ]
                }
            },
            {
                "id": "device_1768527901902_911",
                "type": "router",
                "name": "路由器_8",
                "x": 527,
                "y": 345,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.2.2",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth1",
                            "ip": "192.168.3.1",
                            "subnetMask": "255.255.255.0"
                        }
                    ],
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.2.1"
                        }
                    ]
                }
            },
            {
                "id": "device_1768527901903_722",
                "type": "switch",
                "name": "交换机_9",
                "x": 534,
                "y": 221,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            },
            {
                "id": "device_1768527901904_940",
                "type": "wireless-router",
                "name": "WiFi路由器_10",
                "x": 674,
                "y": 272,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.3.2",
                            "subnetMask": "255.255.255.0",
                            "role": "wan"
                        },
                        {
                            "name": "wlan0",
                            "ip": "192.168.100.1",
                            "subnetMask": "255.255.255.0",
                            "role": "lan"
                        }
                    ],
                    "ssid": "MixNet_WiFi",
                    "security": "open",
                    "password": "",
                    "natEnabled": true,
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.3.1"
                        }
                    ],
                    "dhcpDns": "8.8.8.8"
                }
            },
            {
                "id": "device_1768527901906_332",
                "type": "laptop",
                "name": "笔记本_11",
                "x": 820,
                "y": 283,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.100.146",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.100.1",
                    "dns": "8.8.8.8",
                    "ipMode": "dhcp",
                    "ssid": "MixNet_WiFi",
                    "wirelessEnabled": false,
                    "password": ""
                }
            }
        ],
        "connections": [
            {
                "id": "connection_1",
                "device1Id": "device_1768527901890_891",
                "device2Id": "device_1768527901899_259",
                "isActive": true
            },
            {
                "id": "connection_2",
                "device1Id": "device_1768527901892_861",
                "device2Id": "device_1768527901899_259",
                "isActive": true
            },
            {
                "id": "connection_3",
                "device1Id": "device_1768527901893_879",
                "device2Id": "device_1768527901899_259",
                "isActive": true
            },
            {
                "id": "connection_4",
                "device1Id": "device_1768527901899_259",
                "device2Id": "device_1768527901900_784",
                "isActive": true
            },
            {
                "id": "connection_5",
                "device1Id": "device_1768527901900_784",
                "device2Id": "device_1768527901902_911",
                "isActive": true
            },
            {
                "id": "connection_6",
                "device1Id": "device_1768527901903_722",
                "device2Id": "device_1768527901895_959",
                "isActive": true
            },
            {
                "id": "connection_7",
                "device1Id": "device_1768527901903_722",
                "device2Id": "device_1768527901898_555",
                "isActive": true
            },
            {
                "id": "connection_8",
                "device1Id": "device_1768527901903_722",
                "device2Id": "device_1768527901902_911",
                "isActive": true
            },
            {
                "id": "connection_9",
                "device1Id": "device_1768527901903_722",
                "device2Id": "device_1768527901904_940",
                "isActive": true
            }
        ]
    },
    'complex-network': {
        "devices": [
            {
                "id": "device_1768536428730_482",
                "type": "router",
                "name": "路由器A",
                "x": 224,
                "y": 53,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.1.1",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth1",
                            "ip": "192.168.2.1",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth2",
                            "ip": "192.168.3.1",
                            "subnetMask": "255.255.255.0"
                        }
                    ],
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.3.2"
                        }
                    ]
                }
            },
            {
                "id": "device_1768536431054_377",
                "type": "router",
                "name": "路由器B",
                "x": 448,
                "y": 53,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.3.2",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth1",
                            "ip": "192.168.4.1",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth2",
                            "ip": "192.168.5.1",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth3",
                            "ip": "192.168.4.1",
                            "subnetMask": "255.255.255.0"
                        }
                    ],
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.2.1"
                        },
                        {
                            "network": "192.168.6.0",
                            "subnetMask": "255.255.255.0",
                            "nextHop": "192.168.5.1"
                        }
                    ]
                }
            },
            {
                "id": "device_1768536434390_154",
                "type": "router",
                "name": "路由器C",
                "x": 450,
                "y": 257,
                "width": 80,
                "height": 80,
                "properties": {
                    "interfaces": [
                        {
                            "name": "eth0",
                            "ip": "192.168.5.2",
                            "subnetMask": "255.255.255.0"
                        },
                        {
                            "name": "eth1",
                            "ip": "192.168.6.1",
                            "subnetMask": "255.255.255.0"
                        }
                    ],
                    "routingTable": [
                        {
                            "network": "0.0.0.0",
                            "subnetMask": "0.0.0.0",
                            "nextHop": "192.168.5.1"
                        }
                    ]
                }
            },
            {
                "id": "device_1768536446978_694",
                "type": "computer",
                "name": "PC1",
                "x": 73,
                "y": 65,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.1.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.2.100"
                }
            },
            {
                "id": "device_1768536449031_125",
                "type": "computer",
                "name": "PC3",
                "x": 780,
                "y": 267,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.6.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.6.1",
                    "dns": "192.168.2.100"
                }
            },
            {
                "id": "device_1768536463774_473",
                "type": "computer",
                "name": "PC2",
                "x": 749,
                "y": 66,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.4.100",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.1.1",
                    "dns": "192.168.1.100"
                }
            },
            {
                "id": "device_1768536477774_703",
                "type": "dns-server",
                "name": "DNS服务器",
                "x": 237,
                "y": 249,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.2.100",
                    "subnetMask": "255.255.255.0",
                    "domains": [
                        {
                            "name": "mixly.cn",
                            "ip": "192.168.1.100",
                            "type": "A"
                        },
                        {
                            "name": "mixly.org",
                            "ip": "192.168.4.100",
                            "type": "A"
                        }
                    ],
                    "gateway": "192.168.2.1",
                    "dns": "192.168.2.100"
                }
            },
            {
                "id": "device_1768536898350_928",
                "type": "switch",
                "name": "交换机",
                "x": 607,
                "y": 273,
                "width": 70,
                "height": 50,
                "properties": {
                    "ports": 8,
                    "vlan": {
                        "enabled": false,
                        "vlans": []
                    }
                }
            },
            {
                "id": "device_1768536918978_903",
                "type": "computer",
                "name": "PC4",
                "x": 613,
                "y": 133,
                "width": 60,
                "height": 60,
                "properties": {
                    "ip": "192.168.6.200",
                    "subnetMask": "255.255.255.0",
                    "gateway": "192.168.6.1",
                    "dns": "192.168.1.100"
                }
            }
        ],
        "connections": [
            {
                "id": "connection_3",
                "device1Id": "device_1768536428730_482",
                "device2Id": "device_1768536431054_377",
                "isActive": true
            },
            {
                "id": "connection_4",
                "device1Id": "device_1768536431054_377",
                "device2Id": "device_1768536434390_154",
                "isActive": true
            },
            {
                "id": "connection_5",
                "device1Id": "device_1768536446978_694",
                "device2Id": "device_1768536428730_482",
                "isActive": true
            },
            {
                "id": "connection_7",
                "device1Id": "device_1768536431054_377",
                "device2Id": "device_1768536463774_473",
                "isActive": true
            },
            {
                "id": "connection_8",
                "device1Id": "device_1768536428730_482",
                "device2Id": "device_1768536477774_703",
                "isActive": true
            },
            {
                "id": "connection_9",
                "device1Id": "device_1768536434390_154",
                "device2Id": "device_1768536898350_928",
                "isActive": true
            },
            {
                "id": "connection_10",
                "device1Id": "device_1768536918978_903",
                "device2Id": "device_1768536898350_928",
                "isActive": true
            },
            {
                "id": "connection_11",
                "device1Id": "device_1768536898350_928",
                "device2Id": "device_1768536449031_125",
                "isActive": true
            }
        ]
    }
};

// 加载预设实验
function loadPresetExperiment(experimentType) {
    // 仿真运行时不能加载预设实验
    if (isSimulating) {
        alert('仿真运行时不能加载预设实验，请先停止仿真');
        return;
    }
    
    // 先询问用户是否要覆盖原有拓扑
    if (!confirm('确定要加载预设实验吗？当前拓扑将丢失。')) {
        return;
    }
    
    // 从内置数据中获取实验拓扑
    const topology = presetExperiments[experimentType];
    if (!topology) {
        alert('未知的实验类型');
        log(`未知的实验类型: ${experimentType}`);
        return;
    }
    
    // 直接加载拓扑
    try {
        loadTopology(topology);
        log(`加载预设实验成功: ${experimentType}`);
    } catch (error) {
        alert(`加载预设实验失败: ${error.message}`);
        log(`加载预设实验失败: ${experimentType} - ${error.message}`);
    }
}

// 自动处理DHCP和WiFi配置
function handleAutomaticNetworkConfig() {
    // 遍历所有笔记本电脑
    devices.forEach(device => {
        if (device.type === 'laptop' && device.properties.wirelessEnabled) {
            // 查找匹配的无线路由器（SSID匹配）
            const wirelessRouter = devices.find(router => 
                router.type === 'wireless-router' && 
                router.properties.ssid === device.properties.ssid
            );
            
            if (wirelessRouter) {
                // SSID匹配成功，检查密码是否正确
                let passwordMatch = true;
                const security = wirelessRouter.properties.security ? wirelessRouter.properties.security.toLowerCase() : '';
                if (security && security !== 'open') {
                    // 如果路由器设置了安全模式（非开放），需要验证密码
                    passwordMatch = (device.properties.password === wirelessRouter.properties.password);
                }
                
                if (passwordMatch) {
                    // SSID和密码都匹配成功
                    if (device.properties.ipMode === 'dhcp') {
                        // DHCP模式：从路由器获取IP地址、网关和DNS服务器
                        const routerInterface = wirelessRouter.properties.interfaces.find(iface => iface.name.startsWith('wlan')) || wirelessRouter.properties.interfaces[0];
                        if (routerInterface) {
                            // 从路由器所在网段分配一个IP
                            const routerIP = routerInterface.ip;
                            const subnet = routerIP.substring(0, routerIP.lastIndexOf('.'));
                            // 简单分配一个IP，实际应该检查是否已分配
                            device.properties.ip = `${subnet}.${Math.floor(Math.random() * 254) + 1}`;
                            device.properties.subnetMask = routerInterface.subnetMask;
                            device.properties.gateway = routerIP;
                            // DNS服务器地址来自无线路由器的DHCP设置
                            device.properties.dns = wirelessRouter.properties.dhcpDns || routerIP;
                            log(`笔记本 ${device.name} 通过DHCP从路由器 ${wirelessRouter.name} 获取IP: ${device.properties.ip}, 网关: ${device.properties.gateway}, DNS: ${device.properties.dns}`);
                            
                            // 更新设备IP和网关显示
                            updateDeviceIPDisplay(device);
                        }
                    }
                } else {
                    // 密码不匹配，清除DHCP配置
                    if (device.properties.ipMode === 'dhcp') {
                        device.properties.ip = '';
                        device.properties.subnetMask = '';
                        device.properties.gateway = '';
                        device.properties.dns = '';
                        updateDeviceIPDisplay(device);
                        log(`笔记本 ${device.name} 无法连接到WiFi路由器 ${wirelessRouter.name}：密码错误`);
                    }
                }
            } else {
                // 没有找到匹配的路由器
                if (device.properties.ipMode === 'dhcp') {
                    // 清除DHCP配置（因为路由器不存在）
                    device.properties.ip = '';
                    device.properties.subnetMask = '';
                    device.properties.gateway = '';
                    device.properties.dns = '';
                    updateDeviceIPDisplay(device);
                }
            }
        }
    });
    
    // 更新WiFi连接虚线
    updateWirelessConnections();
}

// 当WiFi路由器被删除或修改时，更新相关笔记本的DHCP配置
function updateLaptopsAfterRouterChange(ssid) {
    if (!ssid) return;
    
    // 查找所有使用此SSID的笔记本
    const affectedLaptops = devices.filter(device => 
        device.type === 'laptop' && 
        device.properties.ipMode === 'dhcp' && 
        device.properties.ssid === ssid
    );
    
    if (affectedLaptops.length === 0) return;
    
    // 查找匹配的WiFi路由器
    const wirelessRouter = devices.find(router => 
        router.type === 'wireless-router' && 
        router.properties.ssid === ssid
    );
    
    if (wirelessRouter) {
        // 路由器存在，检查密码并重新分配IP
        affectedLaptops.forEach(laptop => {
            // 检查密码是否正确
            let passwordMatch = true;
            const security = wirelessRouter.properties.security ? wirelessRouter.properties.security.toLowerCase() : '';
            if (security && security !== 'open') {
                // 如果路由器设置了安全模式（非开放），需要验证密码
                passwordMatch = (laptop.properties.password === wirelessRouter.properties.password);
            }
            
            if (passwordMatch) {
                // 密码正确，分配IP
                const routerInterface = wirelessRouter.properties.interfaces.find(iface => iface.name.startsWith('wlan')) || wirelessRouter.properties.interfaces[0];
                if (routerInterface) {
                    const routerIP = routerInterface.ip;
                    const subnet = routerIP.substring(0, routerIP.lastIndexOf('.'));
                    laptop.properties.ip = `${subnet}.${Math.floor(Math.random() * 254) + 1}`;
                    laptop.properties.subnetMask = routerInterface.subnetMask;
                    laptop.properties.gateway = routerIP;
                    laptop.properties.dns = wirelessRouter.properties.dhcpDns || routerIP;
                    updateDeviceIPDisplay(laptop);
                    log(`笔记本 ${laptop.name} 的DHCP配置已更新（WiFi路由器配置已修改）`);
                }
            } else {
                // 密码错误，清除IP配置
                laptop.properties.ip = '';
                laptop.properties.subnetMask = '';
                laptop.properties.gateway = '';
                laptop.properties.dns = '';
                updateDeviceIPDisplay(laptop);
                log(`笔记本 ${laptop.name} 无法连接到WiFi路由器 ${wirelessRouter.name}：密码错误`);
            }
        });
    } else {
        // 路由器不存在（被删除），清除IP配置
        affectedLaptops.forEach(laptop => {
            laptop.properties.ip = '';
            laptop.properties.subnetMask = '';
            laptop.properties.gateway = '';
            laptop.properties.dns = '';
            updateDeviceIPDisplay(laptop);
            log(`笔记本 ${laptop.name} 的DHCP配置已清除（WiFi路由器已删除）`);
        });
    }
}

// 更新设备的IP和网关显示
function updateDeviceDisplay(device) {
    // 移除现有的IP和网关显示
    const existingIpText = device.element.querySelector('.device-ip');
    if (existingIpText) {
        existingIpText.remove();
    }
    
    const existingGatewayText = device.element.querySelector('.device-gateway');
    if (existingGatewayText) {
        existingGatewayText.remove();
    }
    
    // 根据设备类型获取IP地址和网关
    let ipAddress = "";
    let gatewayAddress = "";
    
    if (device.type === 'computer' || device.type === 'web-server' || device.type === 'dns-server') {
        ipAddress = device.properties.ip;
        gatewayAddress = device.properties.gateway;
    } else if (device.type === 'router' || device.type === 'wireless-router') {
        if (device.properties.interfaces && device.properties.interfaces.length > 0) {
            ipAddress = device.properties.interfaces[0].ip;
            gatewayAddress = ipAddress; // 路由器作为网关
        }
    }
    
    // 只在有IP地址时显示
    if (ipAddress) {
        const deviceData = DEVICE_TYPES[device.type];
        
        const ipText = document.createElementNS("http://www.w3.org/2000/svg", "text");
        ipText.textContent = `IP: ${ipAddress}`;
        ipText.setAttribute("class", "device-ip");
        ipText.setAttribute("x", deviceData.width / 2);
        ipText.setAttribute("y", deviceData.height + 30);
        ipText.setAttribute("text-anchor", "middle");
        ipText.setAttribute("font-size", "10");
        ipText.setAttribute("fill", "#7f8c8d");
        device.element.appendChild(ipText);
        
        // 只在有网关地址时显示（非路由器设备）
        if (gatewayAddress && (device.type === 'computer' || device.type === 'web-server' || device.type === 'dns-server')) {
            const gatewayText = document.createElementNS("http://www.w3.org/2000/svg", "text");
            gatewayText.textContent = `网关: ${gatewayAddress}`;
            gatewayText.setAttribute("class", "device-gateway");
            gatewayText.setAttribute("x", deviceData.width / 2);
            gatewayText.setAttribute("y", deviceData.height + 45);
            gatewayText.setAttribute("text-anchor", "middle");
            gatewayText.setAttribute("font-size", "10");
            gatewayText.setAttribute("fill", "#7f8c8d");
            device.element.appendChild(gatewayText);
        }
    }
}

// 遮罩层点击事件处理函数
function modalClickHandler(e) {
    const modal = document.getElementById('modal');
    // 如果正在拖动，不关闭窗口
    if (isDragging) {
        return;
    }
    // 移除点击遮罩层关闭窗口的功能
    // 保持模态框打开，即使点击了其他地方
}

// 初始化模态对话框
function initModal() {
    const modal = document.getElementById('modal');
    
    // 为所有具有.close类的元素添加事件监听（包括动态添加的）
    modal.addEventListener('click', (e) => {
        if (e.target.classList.contains('close')) {
            modal.style.display = 'none';
        }
    });
    
    // 添加遮罩层点击关闭功能
    window.addEventListener('click', modalClickHandler);
}

// 显示Ping工具
function showPingTool() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    // 过滤掉路由器和交换机类型的设备
    const availableDevices = devices.filter(device => device.type !== 'router' && device.type !== 'wireless-router' && device.type !== 'switch');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>网络联通测试工具</h3>
        </div>
        <div class="property-form">
            <div class="form-group">
                <label for="ping-source">源设备</label>
                <select id="ping-source">
                    ${availableDevices.map(device => `<option value="${device.id}">${device.name}</option>`).join('')}
                </select>
            </div>
            <div class="form-group">
                <label for="ping-destination">目标主机 (IP或域名)</label>
                <input type="text" id="ping-destination" placeholder="192.168.1.1 或 example.com">
            </div>
            <div class="form-group">
                <label for="ping-count">发送次数</label>
                <input type="number" id="ping-count" value="4" min="1" max="10">
            </div>
            <div class="form-actions">
                <button class="btn-save" onclick="executePing()">开始Ping</button>
                <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">取消</button>
            </div>
            
            <!-- Ping结果显示区域 -->
            <div class="ping-results" style="margin-top: 10px; padding: 15px; background-color: #f8f9fa; border-radius: 5px; font-family: 'Courier New', monospace; font-size: 12px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;">
                <div id="ping-output" style="color: #2c3e50;"></div>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
}

// 显示网络命令测试工具
function showNetworkCommandTool() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    // 获取所有可用设备（包括服务器，但排除路由器和交换机）
    const allDevices = devices.filter(device => device.type !== 'router' && device.type !== 'wireless-router' && device.type !== 'switch');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>网络命令测试工具</h3>
        </div>
        <div class="property-form">
            <div class="form-group">
                <label for="network-command-device">源设备</label>
                <select id="network-command-device">
                    ${allDevices.map(device => `<option value="${device.id}">${device.name}</option>`).join('')}
                </select>
            </div>
            <div class="form-group">
                <label for="network-command-os">操作系统</label>
                <select id="network-command-os">
                    <option value="windows">Windows</option>
                    <option value="linux">Linux</option>
                </select>
            </div>
            
            <!-- 支持的命令列表（按操作系统显示） -->
            <div id="command-list" style="margin-bottom: 10px; padding: 8px; background-color: #f8f9fa; border-radius: 4px; font-size: 14px;">
                <div id="windows-commands" style="display: none;">
                    支持的命令：arp, ipconfig, ping, tracert, nslookup, route
                </div>
                <div id="linux-commands" style="display: none;">
                    支持的命令：arp, ifconfig, ping, traceroute, nslookup, route
                </div>
            </div>
            
            <!-- 命令终端（更接近真实终端） -->
            <div style="border: 1px solid #333; border-radius: 4px; background-color: #000; color: #fff; padding: 5px; font-family: 'Courier New', monospace; font-size: 12px; height: 350px; overflow-y: auto;">
                <div id="command-terminal" style="line-height: 1.4;">
                    <!-- 命令和输出将动态显示在这里 -->
                </div>
                <!-- 动态输入行 -->
                <div id="input-line" style="display: flex; align-items: center; margin-top: 2px; position: relative; z-index: 10;">
                    <span id="command-prompt" style="color: #0f0; margin-right: 4px;">$</span>
                    <input type="text" id="command-input" style="flex: 1; background-color: transparent; border: none; color: #fff; outline: none; font-family: 'Courier New', monospace; font-size: 12px; cursor: text; pointer-events: auto; z-index: 11;">
                </div>
            </div>
            

            
            <div class="form-actions">
                <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">关闭</button>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
    
    // 初始化命令终端
    initCommandTerminal();
}

// 全局变量：当前命令状态
let commandState = {
    mode: 'normal', // normal, nslookup
    nslookup: {
        server: null,
        prompt: '> '
    }
};

// 命令历史记录
let commandHistory = [];
let historyIndex = -1;

// 初始化命令终端
function initCommandTerminal() {
    const terminal = document.getElementById('command-terminal');
    const commandInput = document.getElementById('command-input');
    const commandPrompt = document.getElementById('command-prompt');
    const selectedOS = document.getElementById('network-command-os').value;
    
    // 重置命令状态
    commandState = {
        mode: 'normal',
        nslookup: {
            server: null,
            prompt: '> '
        }
    };
    
    // 清空终端，不显示欢迎信息
    terminal.innerHTML = '';
    
    // 设置命令提示符和显示命令列表
    setCommandPrompt(selectedOS);
    
    // 显示对应操作系统的命令列表
    const windowsCommands = document.getElementById('windows-commands');
    const linuxCommands = document.getElementById('linux-commands');
    if (selectedOS === 'windows') {
        windowsCommands.style.display = 'block';
        linuxCommands.style.display = 'none';
    } else {
        windowsCommands.style.display = 'none';
        linuxCommands.style.display = 'block';
    }
    
    // 移除现有的事件监听器，避免重复绑定
    commandInput.replaceWith(commandInput.cloneNode(true));
    
    // 重新获取输入框引用
    const newCommandInput = document.getElementById('command-input');
    
    // 命令输入事件处理
    newCommandInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            const command = this.value.trim();
            if (command) {
                executeCommand(command);
                
                // 添加到命令历史
                if (commandHistory[commandHistory.length - 1] !== command) {
                    commandHistory.push(command);
                }
                
                // 重置历史索引
                historyIndex = commandHistory.length;
                
                this.value = '';
                
                // 滚动到底部
                const terminalContainer = terminal.parentElement;
                terminalContainer.scrollTop = terminalContainer.scrollHeight;
            }
        } else if (e.key === 'ArrowUp') {
            // 向上箭头：浏览命令历史
            e.preventDefault();
            if (historyIndex > 0) {
                historyIndex--;
                this.value = commandHistory[historyIndex] || '';
                // 光标移动到末尾
                this.setSelectionRange(this.value.length, this.value.length);
            }
        } else if (e.key === 'ArrowDown') {
            // 向下箭头：浏览命令历史
            e.preventDefault();
            if (historyIndex < commandHistory.length - 1) {
                historyIndex++;
                this.value = commandHistory[historyIndex] || '';
                // 光标移动到末尾
                this.setSelectionRange(this.value.length, this.value.length);
            } else {
                historyIndex = commandHistory.length;
                this.value = '';
                // 光标移动到末尾
                this.setSelectionRange(0, 0);
            }
        } else if (e.ctrlKey && e.key === 'l') {
            // Ctrl+L：清屏
            e.preventDefault();
            terminal.innerHTML = '';
        }
    });
    
    // 确保输入框可以获取焦点
    newCommandInput.style.pointerEvents = 'auto';
    newCommandInput.style.zIndex = '1000';
    newCommandInput.style.position = 'relative';
    
    // 修复光标位置
    newCommandInput.addEventListener('focus', function() {
        // 光标移动到末尾
        this.setSelectionRange(this.value.length, this.value.length);
    });
    
    // 确保点击终端区域时输入框获得焦点
    terminal.parentElement.addEventListener('click', function() {
        newCommandInput.focus();
    });
    
    // 初始焦点
    setTimeout(() => {
        newCommandInput.focus();
    }, 100);
    
    // 点击终端区域时，输入框获得焦点
    terminal.addEventListener('click', () => {
        newCommandInput.focus();
    });
    
    // 监听操作系统变化事件
    document.getElementById('network-command-os').addEventListener('change', (e) => {
        setCommandPrompt(e.target.value);
        
        // 更新命令列表显示
        const selectedOS = e.target.value;
        const windowsCommands = document.getElementById('windows-commands');
        const linuxCommands = document.getElementById('linux-commands');
        if (selectedOS === 'windows') {
            windowsCommands.style.display = 'block';
            linuxCommands.style.display = 'none';
        } else {
            windowsCommands.style.display = 'none';
            linuxCommands.style.display = 'block';
        }
        
        // 确保输入框保持焦点
        const currentCommandInput = document.getElementById('command-input');
        if (currentCommandInput) {
            currentCommandInput.focus();
        }
    });
}

// 设置命令提示符
function setCommandPrompt(os) {
    const commandPrompt = document.getElementById('command-prompt');
    
    if (os === 'windows') {
        commandPrompt.textContent = 'C:\>';
    } else {
        commandPrompt.textContent = '$';
    }
}

// 执行命令
function executeCommand(command) {
    // 检查是否正在仿真
    if (!isSimulating) {
        const terminal = document.getElementById('command-terminal');
        terminal.innerHTML += `<div style="color: #e74c3c;">错误：请先启动仿真再执行命令</div>`;
        return;
    }
    
    const terminal = document.getElementById('command-terminal');
    const selectedDeviceId = document.getElementById('network-command-device').value;
    const selectedOS = document.getElementById('network-command-os').value;
    const commandPrompt = document.getElementById('command-prompt');
    
    // 获取选中的设备
    const selectedDevice = devices.find(device => device.id === selectedDeviceId);
    
    // 显示输入的命令和提示符
    const currentPrompt = commandPrompt.textContent;
    terminal.innerHTML += `<div style="margin: 2px 0;"><span style="color: #0f0;">${currentPrompt}</span> <span>${command}</span></div>`;
    
    // 根据当前命令模式处理
    if (commandState.mode === 'nslookup') {
        // 处理nslookup交互式命令
        const output = handleNSLookupInteractive(command, selectedDevice, selectedOS);
        if (output !== null) {
            terminal.innerHTML += `<div style="margin: 2px 0; white-space: pre-wrap;">${output}</div>`;
        }
    } else {
        // 正常模式：解析命令和参数
        const parts = command.split(/\s+/);
        const cmd = parts[0].toLowerCase();
        const params = parts.slice(1);
        
        // 根据操作系统执行相应的命令
        let output = '';
        if (selectedOS === 'windows') {
            output = executeWindowsCommand(cmd, params, selectedDevice);
        } else {
            output = executeLinuxCommand(cmd, params, selectedDevice);
        }
        
        // 显示命令输出
        terminal.innerHTML += `<div style="margin: 2px 0; white-space: pre-wrap;">${output}</div>`;
    }
    
    // 滚动到底部
    const terminalContainer = terminal.parentElement;
    terminalContainer.scrollTop = terminalContainer.scrollHeight;
    
    // 确保输入框保持焦点
    const currentCommandInput = document.getElementById('command-input');
    if (currentCommandInput) {
        currentCommandInput.focus();
    }
}

// 执行Windows命令
function executeWindowsCommand(cmd, params, device) {
    // 使用统一的IP获取函数，确保正确处理DHCP模式
    const ip = getDeviceIp(device);
    const subnetMask = device.properties.subnetMask || '255.255.255.0';
    const gateway = device.properties.gateway || '';
    const dns = device.properties.dns || '';
    
    // 如果无法获取IP（如DHCP模式未分配），返回错误信息
    if (!ip && (cmd === 'ipconfig' || cmd === 'arp' || cmd === 'ping' || cmd === 'tracert')) {
        if (device.type === 'laptop' && device.properties.ipMode === 'dhcp') {
            return `错误：设备使用DHCP模式但尚未连接到WiFi路由器，无法获取IP地址。\n请确保设备已连接到匹配的WiFi路由器。`;
        }
        return `错误：设备未配置IP地址。`;
    }
    
    switch (cmd) {
        case 'arp':
            return `接口: ${ip} --- 0x10003

  Internet 地址         物理地址              类型
  ${gateway}          00-11-22-33-44-55     动态
  ${getNetworkAddress(ip, subnetMask)}  ff-ff-ff-ff-ff-ff     静态`;
            
        case 'ipconfig':
            return `Windows IP 配置


以太网适配器 本地连接:

   连接特定的 DNS 后缀 . . . . . . . :
   本地链接 IPv6 地址. . . . . . . . : fe80::1111:2222:3333:4444%10
   IPv4 地址 . . . . . . . . . . . . : ${ip}
   子网掩码  . . . . . . . . . . . . : ${subnetMask}
   默认网关. . . . . . . . . . . . . : ${gateway}`;
            
        case 'ping':
            const target = params[0] || 'localhost';
            let ipAddress = '127.0.0.1'; // 默认localhost IP
            
            // 如果目标是域名（包含.且不是IPv4地址），获取仿真IP
            if (target.includes('.') && !/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(target)) {
                ipAddress = getSimulatedIpForDomain(target, device);
            } else if (target !== 'localhost') {
                ipAddress = target; // 如果是IP地址，直接使用
            }
            
            return `正在 Ping ${target} [${ipAddress}] 具有 32 字节的数据:
来自 ${ipAddress} 的回复: 字节=32 时间=1ms TTL=128
来自 ${ipAddress} 的回复: 字节=32 时间=2ms TTL=128
来自 ${ipAddress} 的回复: 字节=32 时间=1ms TTL=128
来自 ${ipAddress} 的回复: 字节=32 时间=3ms TTL=128

${target} 的 Ping 统计信息:
    数据包: 已发送 = 4，已接收 = 4，丢失 = 0 (0% 丢失)，
往返行程的估计时间(以毫秒为单位):
    最短 = 1ms，最长 = 3ms，平均 = 1ms`;
            
        case 'tracert':
            const tracertTarget = params[0] || 'example.com';
            let tracertIpAddress = '93.184.216.34'; // 默认IP
            
            // 如果目标是域名（包含.且不是IPv4地址），获取仿真IP
            if (tracertTarget.includes('.') && !/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(tracertTarget)) {
                tracertIpAddress = getSimulatedIpForDomain(tracertTarget, device);
            } else if (tracertTarget !== 'localhost') {
                tracertIpAddress = tracertTarget; // 如果是IP地址，直接使用
            }
            
            return `通过最多 30 个跃点跟踪到 ${tracertTarget} [${tracertIpAddress}] 的路由:

  1     1 ms    <1 ms    <1 ms  ${gateway}
  2    10 ms     8 ms     9 ms  10.0.0.1
  3    25 ms    22 ms    23 ms  202.100.1.1
  4    35 ms    33 ms    34 ms  ${tracertIpAddress}

跟踪完成。`;
            
        case 'nslookup':
            const nslookupTarget = params[0] || null;
            
            // 进入nslookup交互式模式
            commandState.mode = 'nslookup';
            commandState.nslookup.server = dns;
            safeSetTextContent('command-prompt', 'nslookup>');
            
            // 如果有参数，直接查询
            if (nslookupTarget) {
                return executeNSLookupQuery(nslookupTarget, dns, device, 'windows');
            }
            
            return `默认服务器:  UnKnown
Address:  ${dns}

nslookup> `;
            
        case 'route':
            return `==========================================================================
接口列表
 10...00 11 22 33 44 55 ......Intel(R) Ethernet Connection
==========================================================================

IPv4 路由表
==========================================================================
活动路由:
网络目标        网络掩码          网关       接口   跃点数
          0.0.0.0          0.0.0.0      ${gateway}     ${ip}     25
        127.0.0.0        255.0.0.0         在链路上         127.0.0.1    331
        127.0.0.1  255.255.255.255         在链路上         127.0.0.1    331
  127.255.255.255  255.255.255.255         在链路上         127.0.0.1    331
        224.0.0.0        240.0.0.0         在链路上         127.0.0.1    331
        224.0.0.0        240.0.0.0         在链路上      ${ip}     276
  255.255.255.255  255.255.255.255         在链路上         127.0.0.1    331
  255.255.255.255  255.255.255.255         在链路上      ${ip}     276
==========================================================================`;
            
        default:
            return `'${cmd}' 不是内部或外部命令，也不是可运行的程序或批处理文件。`;
    }
}

// 执行Linux命令
function executeLinuxCommand(cmd, params, device) {
    // 使用统一的IP获取函数，确保正确处理DHCP模式
    const ip = getDeviceIp(device);
    const subnetMask = device.properties.subnetMask || '255.255.255.0';
    const gateway = device.properties.gateway || '';
    const dns = device.properties.dns || '';
    
    // 如果无法获取IP（如DHCP模式未分配），返回错误信息
    if (!ip && (cmd === 'ifconfig' || cmd === 'arp' || cmd === 'ping' || cmd === 'traceroute')) {
        if (device.type === 'laptop' && device.properties.ipMode === 'dhcp') {
            return `错误：设备使用DHCP模式但尚未连接到WiFi路由器，无法获取IP地址。\n请确保设备已连接到匹配的WiFi路由器。`;
        }
        return `错误：设备未配置IP地址。`;
    }
    
    switch (cmd) {
        case 'arp':
            return `Address                  HWtype  HWaddress           Flags Mask            Iface
${gateway}        ether   00:11:22:33:44:55   C                     eth0
${getNetworkAddress(ip, subnetMask)}  ether   ff:ff:ff:ff:ff:ff   C                     eth0`;
            
        case 'ifconfig':
            return `eth0: flags=4163<UP,BROADCAST,RUNNING,MULTICAST>  mtu 1500
        inet ${ip}  netmask ${subnetMask}  broadcast ${getBroadcastAddress(ip, subnetMask)}
        inet6 fe80::1111:2222:3333:4444  prefixlen 64  scopeid 0x20<link>
        ether 00:11:22:33:44:55  txqueuelen 1000  (Ethernet)
        RX packets 0  bytes 0 (0.0 B)
        RX errors 0  dropped 0  overruns 0  frame 0
        TX packets 0  bytes 0 (0.0 B)
        TX errors 0  dropped 0 overruns 0  carrier 0  collisions 0

lo: flags=73<UP,LOOPBACK,RUNNING>  mtu 65536
        inet 127.0.0.1  netmask 255.0.0.0
        inet6 ::1  prefixlen 128  scopeid 0x10<host>
        loop  txqueuelen 1000  (Local Loopback)
        RX packets 0  bytes 0 (0.0 B)
        RX errors 0  dropped 0  overruns 0  frame 0
        TX packets 0  bytes 0 (0.0 B)
        TX errors 0  dropped 0 overruns 0  carrier 0  collisions 0`;
            
        case 'ping':
            const target = params[0] || 'localhost';
            let ipAddress = '127.0.0.1'; // 默认localhost IP
            
            // 如果目标是域名（包含.且不是IPv4地址），获取仿真IP
            if (target.includes('.') && !/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(target)) {
                ipAddress = getSimulatedIpForDomain(target, device);
            } else if (target !== 'localhost') {
                ipAddress = target; // 如果是IP地址，直接使用
            }
            
            return `PING ${target} (${ipAddress}) 56(84) bytes of data.
64 bytes from ${ipAddress} (${ipAddress}): icmp_seq=1 ttl=64 time=0.31 ms
64 bytes from ${ipAddress} (${ipAddress}): icmp_seq=2 ttl=64 time=0.28 ms
64 bytes from ${ipAddress} (${ipAddress}): icmp_seq=3 ttl=64 time=0.30 ms
64 bytes from ${ipAddress} (${ipAddress}): icmp_seq=4 ttl=64 time=0.29 ms

--- ${target} ping statistics ---
4 packets transmitted, 4 received, 0% packet loss, time 3004ms
rtt min/avg/max/mdev = 0.28/0.295/0.31/0.014 ms`;
            
        case 'traceroute':
            const tracerouteTarget = params[0] || 'example.com';
            let tracerouteIpAddress = '93.184.216.34'; // 默认IP
            
            // 如果目标是域名（包含.且不是IPv4地址），获取仿真IP
            if (tracerouteTarget.includes('.') && !/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(tracerouteTarget)) {
                tracerouteIpAddress = getSimulatedIpForDomain(tracerouteTarget, device);
            } else if (tracerouteTarget !== 'localhost') {
                tracerouteIpAddress = tracerouteTarget; // 如果是IP地址，直接使用
            }
            
            return `traceroute to ${tracerouteTarget} (${tracerouteIpAddress}), 30 hops max, 60 byte packets
 1  ${gateway} (${gateway})  1.234 ms  1.123 ms  1.098 ms
 2  10.0.0.1 (10.0.0.1)  10.456 ms  10.345 ms  10.234 ms
 3  202.100.1.1 (202.100.1.1)  25.678 ms  25.567 ms  25.456 ms
 4  ${tracerouteIpAddress} (${tracerouteIpAddress})  35.789 ms  35.678 ms  35.567 ms`;
            
        case 'nslookup':
            const nslookupTarget = params[0] || null;
            
            // 进入nslookup交互式模式
            commandState.mode = 'nslookup';
            commandState.nslookup.server = dns;
            safeSetTextContent('command-prompt', 'nslookup>');
            
            // 如果有参数，直接查询
            if (nslookupTarget) {
                return executeNSLookupQuery(nslookupTarget, dns, device, 'linux');
            }
            
            return `Server:         ${dns}
Address:        ${dns}#53

nslookup> `;
            
        case 'route':
            return `Kernel IP routing table
Destination     Gateway         Genmask         Flags Metric Ref    Use Iface
0.0.0.0         ${gateway}      0.0.0.0         UG    0      0        0 eth0
127.0.0.0       0.0.0.0         255.0.0.0       U     0      0        0 lo
${getNetworkAddress(ip, subnetMask)}  0.0.0.0         ${subnetMask}     U     0      0        0 eth0`;
            
        default:
            return `bash: ${cmd}: command not found`;
    }
}

// 辅助函数：获取网络地址
function getNetworkAddress(ip, subnetMask) {
    const ipParts = ip.split('.').map(Number);
    const maskParts = subnetMask.split('.').map(Number);
    const networkParts = ipParts.map((part, index) => part & maskParts[index]);
    return networkParts.join('.');
}

// 辅助函数：获取广播地址
function getBroadcastAddress(ip, subnetMask) {
    const ipParts = ip.split('.').map(Number);
    const maskParts = subnetMask.split('.').map(Number);
    const notMaskParts = maskParts.map(part => 255 - part);
    const broadcastParts = ipParts.map((part, index) => part | notMaskParts[index]);
    return broadcastParts.join('.');
}

// DNS域名解析函数
function resolveDomain(domain, recordType) {
    let dnsServers = [];
    let records = [];
    let serverIp = '';
    
    // 获取所有DNS服务器
    dnsServers = devices.filter(device => device.type === 'dns-server');
    
    // 尝试在DNS服务器上查找域名
    for (const dnsServer of dnsServers) {
        if (dnsServer.type === 'dns-server' && dnsServer.properties.domains) {
            serverIp = dnsServer.properties.ip;
            for (const domainRecord of dnsServer.properties.domains) {
                if (domain === domainRecord.name && (recordType === 'ANY' || domainRecord.type === recordType)) {
                    records.push({
                        name: domainRecord.name,
                        type: domainRecord.type,
                        value: domainRecord.ip
                    });
                }
            }
            // 如果找到记录，停止查找
            if (records.length > 0) {
                break;
            }
        }
    }
    
    return {
        success: records.length > 0,
        records: records,
        serverIp: serverIp
    };
}

// 获取域名对应的真实IP地址（基于DNS服务器配置）
function getSimulatedIpForDomain(domain, device) {
    // 使用resolveDomain函数从DNS服务器配置中获取真实的IP地址
    const resolved = resolveDomain(domain, 'A');
    
    if (resolved.success && resolved.records.length > 0) {
        // 返回第一个匹配的IP地址
        return resolved.records[0].value;
    }
    
    // 如果DNS查询失败，返回默认IP（设备网段内）
    const deviceIpParts = device.properties.ip.split('.');
    const networkPrefix = deviceIpParts.slice(0, 3).join('.');
    return `${networkPrefix}.254`;
}

// 执行nslookup查询（基于DNS服务器配置）
function executeNSLookupQuery(target, server, device, os, recordType = 'A') {
    // 检查源设备是否能到达DNS服务器
    if (server) {
        const dnsServerIp = server;
        if (!isReachable(device, dnsServerIp)) {
            if (os === 'windows') {
                return `服务器:  UnKnown\nAddress:  ${server}\n\n*** UnKnown 找不到 ${target}: Timed out`;
            } else {
                return `;; connection timed out; no servers could be reached`;
            }
        }
    }
    
    // 使用resolveDomain函数从DNS服务器配置中获取真实的IP地址
    const resolved = resolveDomain(target, recordType);
    let results = [];
    
    if (resolved.success && resolved.records.length > 0) {
        // 获取所有匹配的记录
        results = resolved.records;
    }
    
    // DNS服务器名称：通常显示为UnKnown（因为没有反向DNS记录）
    const serverName = 'UnKnown';
    
    if (os === 'windows') {
        if (results.length > 0) {
            let output = `服务器:  ${serverName}\nAddress:  ${server}\n\n非权威应答:\n名称:    ${target}\n`;
            
            if (recordType === 'MX') {
                results.forEach((record, index) => {
                    output += `MX preference = ${index + 10}, mail exchanger = ${record.value}\n`;
                });
            } else {
                const values = results.map(record => record.value);
                output += `Addresses:  ${values.join(', ')}\n`;
            }
            return output;
        } else {
            // 找不到域名时，不显示IP地址，直接显示找不到
            return `服务器:  ${serverName}\nAddress:  ${server}\n\n*** ${serverName} 找不到 ${target}: Non-existent domain`;
        }
    } else {
        // Linux系统显示IP地址，不显示服务器名称
        if (results.length > 0) {
            let output = `Server:         ${server}\nAddress:        ${server}#53\n\nNon-authoritative answer:\nName:   ${target}\n`;
            
            if (recordType === 'MX') {
                results.forEach((record, index) => {
                    output += `MX preference = ${index + 10}, mail exchanger = ${record.value}\n`;
                });
            } else {
                results.forEach(record => {
                    output += `Address:   ${record.value}\n`;
                });
            }
            return output;
        } else {
            // 找不到域名时，不显示IP地址，直接显示找不到
            return `Server:         ${server}\nAddress:        ${server}#53\n\n** server can't find ${target}: NXDOMAIN`;
        }
    }
}

// 处理nslookup交互式命令
function handleNSLookupInteractive(command, device, os) {
    const parts = command.split(/\s+/);
    const cmd = parts[0].toLowerCase();
    const params = parts.slice(1);
    const commandPrompt = document.getElementById('command-prompt');
    const dns = device.properties.dns;
    
    switch (cmd) {
        case 'exit':
        case 'quit':
            // 退出nslookup模式
            commandState.mode = 'normal';
            setCommandPrompt(os);
            return '退出 nslookup';
            
        case 'server':
            // 切换DNS服务器
            if (params.length > 0) {
                commandState.nslookup.server = params[0];
                return `服务器:  ${params[0]}\nAddress:  ${params[0]}`;
            }
            return '请指定DNS服务器地址';
            
        case 'set':
            // 设置nslookup选项
            if (params.length > 0) {
                const option = params[0].toLowerCase();
                if (option === 'type') {
                    if (params.length > 1) {
                        commandState.nslookup.recordType = params[1].toUpperCase();
                        return `默认查询类型为: ${params[1].toUpperCase()}`;
                    }
                }
            }
            return '设置功能未完全实现';
            
        default:
            // 处理域名查询
            if (cmd) {
                const recordType = commandState.nslookup.recordType || 'A';
                return executeNSLookupQuery(cmd, commandState.nslookup.server || dns, device, os, recordType);
            }
            return null;
    }
}

// 执行Ping测试
function executePing() {
    const sourceId = document.getElementById('ping-source').value;
    const destination = document.getElementById('ping-destination').value;
    const count = parseInt(document.getElementById('ping-count').value);
    
    const source = devices.find(device => device.id === sourceId);
    if (source) {
        const pingOutput = document.getElementById('ping-output');
        pingOutput.innerHTML = ''; // 清空之前的结果
        
        // 记录日志
        log(`开始Ping测试: ${source.name} → ${destination}`);
        
        // 获取源设备的IP地址（使用统一的获取函数）
        let sourceIp = getDeviceIp(source);
        if (!sourceIp) {
            // 如果是笔记本且DHCP模式未分配IP，提示用户
            if (source.type === 'laptop' && source.properties.ipMode === 'dhcp') {
                pingOutput.innerHTML = `<span style="color: #e74c3c;">错误：设备 ${source.name} 使用DHCP模式但尚未连接到WiFi路由器，无法获取IP地址。请确保设备已连接到匹配的WiFi路由器。</span>`;
                return;
            }
            sourceIp = "未知IP";
        }
        
        // 模拟真实Ping命令的开始输出
        const startTime = new Date();
        pingOutput.innerHTML += `正在 Ping ${destination} 来自 ${sourceIp}:

`;
        
        // 统计信息
        let sent = count;
        let received = 0;
        let minTime = Infinity;
        let maxTime = 0;
        let totalTime = 0;
        
        // 解析目标地址（支持域名）
        let targetIp = destination;
        let isDomain = false;
        
        // 检查是否为域名（简单判断：包含点且不是IP地址格式）
        if (destination.includes('.') && !/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(destination)) {
            isDomain = true;
            // 尝试DNS解析
            let dnsServers = [];
            let resolved = false;
            
            // 首先获取源设备配置的DNS服务器
            const sourceDns = source.properties.dns;
            if (sourceDns) {
                // 查找配置的DNS服务器设备
                const configuredDnsServer = devices.find(device => 
                    (device.type === 'dns-server' || device.type === 'wireless-router') && 
                    getDeviceIp(device) === sourceDns
                );
                if (configuredDnsServer) {
                    dnsServers.push(configuredDnsServer);
                }
            }
            
            // 如果没有配置DNS服务器或配置的DNS服务器不可用，使用所有可用的DNS服务器
            if (dnsServers.length === 0) {
                dnsServers = devices.filter(device => device.type === 'dns-server');
            }
            
            // 尝试在DNS服务器上查找域名
            for (const dnsServer of dnsServers) {
                if (dnsServer.type === 'dns-server' && dnsServer.properties.domains) {
                    for (const domain of dnsServer.properties.domains) {
                        if (destination === domain.name) {
                            targetIp = domain.ip;
                            resolved = true;
                            break;
                        } else if (destination.endsWith(domain.name) && domain.type === 'MX') {
                            // MX记录匹配：当邮件地址域名部分与MX记录域名匹配时
                            targetIp = domain.ip;
                            resolved = true;
                            break;
                        }
                    }
                } else if (dnsServer.type === 'wireless-router' && dnsServer.properties.dhcpDns) {
                    // 无线路由器作为DNS中继，尝试使用它配置的DNS服务器
                    const upstreamDnsServer = devices.find(device => 
                        device.type === 'dns-server' && 
                        getDeviceIp(device) === dnsServer.properties.dhcpDns
                    );
                    if (upstreamDnsServer && upstreamDnsServer.properties.domains) {
                        for (const domain of upstreamDnsServer.properties.domains) {
                            if (destination === domain.name) {
                                targetIp = domain.ip;
                                resolved = true;
                                break;
                            } else if (destination.endsWith(domain.name) && domain.type === 'MX') {
                                // MX记录匹配：当邮件地址域名部分与MX记录域名匹配时
                                targetIp = domain.ip;
                                resolved = true;
                                break;
                            }
                        }
                    }
                }
                if (resolved) break;
            }
            
            if (!resolved) {
                pingOutput.innerHTML += `Ping 请求找不到主机 ${destination}。请检查该名称，然后重试。

`;
                log(`Ping ${destination} 来自 ${source.name}: DNS解析失败`);
                
                // 显示统计信息
                const endTime = new Date();
                const elapsed = Math.floor((endTime - startTime) / 1000);
                
                pingOutput.innerHTML += `${destination} 的 Ping 统计信息:
    数据包: 已发送 = 0, 已接收 = 0, 丢失 = 0 (0% 丢失),
往返行程的估计时间(以毫秒为单位):
    最短 = 0ms, 最长 = 0ms, 平均 = 0ms
`;
                
                return; // 直接返回，不进行后续的ping模拟
            }
            
            // 显示DNS解析结果
            pingOutput.innerHTML += `[DNS 解析]: ${destination} → ${targetIp}

`;
        }
        
        // 查找目标设备信息
        const targetInfo = findDeviceByIp(targetIp);
        
        // 首先检查目标IP是否存在
        if (!targetInfo) {
            pingOutput.innerHTML += `Ping 请求找不到主机 ${destination}。请检查该名称，然后重试。\n\n`;
            log(`Ping ${destination} 来自 ${source.name}: 找不到主机`);
            
            // 显示统计信息
            const endTime = new Date();
            const elapsed = Math.floor((endTime - startTime) / 1000);
            
            pingOutput.innerHTML += `${destination} 的 Ping 统计信息:\n    数据包: 已发送 = 0, 已接收 = 0, 丢失 = 0 (0% 丢失),\n往返行程的估计时间(以毫秒为单位):\n    最短 = 0ms, 最长 = 0ms, 平均 = 0ms\n`;
            
            return; // 直接返回，不进行后续的ping模拟
        }
        
        // 检查网络可达性 - 严格使用isReachable函数
        let isNetworkReachable = isReachable(source, targetIp);
        
        // 计算路径长度（跳数）
        let pathLength = 1; // 默认直接连接
        if (targetInfo.device !== source) {
            const path = findPath(source, targetInfo.device);
            pathLength = path ? path.length - 1 : 1;
        }
        
        // 根据网络拓扑计算基础成功率
        let baseSuccessRate = 0.95; // 直接连接的成功率
        if (!isNetworkReachable) {
            baseSuccessRate = 0.0; // 不可达网络成功率为0
        } else if (pathLength > 1) {
            // 每经过一个路由器，成功率降低一些
            baseSuccessRate = Math.max(0.7, 1 - (pathLength - 1) * 0.05);
        }
        
        // 基础延迟（毫秒）
        let baseDelay = 10; // 直接连接的基础延迟
        if (pathLength > 1) {
            // 每经过一个路由器，增加延迟
            baseDelay = baseDelay + (pathLength - 1) * 15;
        }
        
        // 模拟Ping结果
        for (let i = 0; i < count; i++) {
            setTimeout(() => {
                // 根据网络状态计算本次ping的成功率
                const success = Math.random() < baseSuccessRate;
                
                if (success) {
                    received++;
                    // 延迟包含基础延迟和随机波动
                    const time = baseDelay + Math.floor(Math.random() * 30);
                    minTime = Math.min(minTime, time);
                    maxTime = Math.max(maxTime, time);
                    totalTime += time;
                    
                    // 计算TTL值（根据跳数）
                    const ttl = 64 - pathLength;
                    
                    // 在响应中使用解析后的IP地址，如果是域名则同时显示域名和IP
                    const responseDestination = isDomain ? `${destination} [${targetIp}]` : targetIp;
                    pingOutput.innerHTML += `来自 ${responseDestination} 的回复: 字节=32 时间=${time}ms TTL=${ttl}
`;
                    log(`Ping ${destination} 来自 ${source.name}: 时间=${time}ms`);
                    
                    // 显示数据包动画
                    if (targetInfo.device && targetInfo.device !== source) {
                        const path = findPath(source, targetInfo.device);
                        if (path) {
                            createPacketAlongPath(path, source, targetInfo.device);
                        }
                    }
                } else {
                    let errorMsg = "请求超时。";
                    if (!isNetworkReachable) {
                        errorMsg = "目标主机不可达。";
                    }
                    pingOutput.innerHTML += `${errorMsg}
`;
                    log(`Ping ${destination} 来自 ${source.name}: ${errorMsg.trim()}`);
                }
                
                // 所有Ping完成后显示统计信息
                if (i === count - 1) {
                    setTimeout(() => {
                        const endTime = new Date();
                        const elapsed = Math.floor((endTime - startTime) / 1000);
                        const avgTime = received > 0 ? Math.round(totalTime / received) : 0;
                        
                        pingOutput.innerHTML += `
${destination} 的 Ping 统计信息:
    数据包: 已发送 = ${sent}, 已接收 = ${received}, 丢失 = ${sent - received} (${Math.round((sent - received) / sent * 100)}% 丢失),
往返行程的估计时间(以毫秒为单位):
    最短 = ${minTime === Infinity ? 0 : minTime}ms, 最长 = ${maxTime}ms, 平均 = ${avgTime}ms
    路径跳数: ${pathLength}
`;
                        
                        log(`Ping测试完成: ${source.name} → ${destination}, 结果: ${received}/${sent} 响应, 平均时间=${avgTime}ms, 路径跳数=${pathLength}`);
                    }, 200);
                }
            }, i * 1000);
        }
    }
}

// 显示Traceroute工具
function showTracerouteTool() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>数据跟踪测试工具</h3>
        </div>
        <div class="property-form">
            <div class="form-group">
                <label for="traceroute-source">源设备</label>
                <select id="traceroute-source">
                    ${devices.map(device => `<option value="${device.id}">${device.name}</option>`).join('')}
                </select>
            </div>
            <div class="form-group">
                <label for="traceroute-destination">目标IP</label>
                <input type="text" id="traceroute-destination" placeholder="10.0.0.100">
            </div>
            <div class="form-group">
                <label>结果输出</label>
                <div id="traceroute-result" style="border: 1px solid #ccc; padding: 10px; height: 200px; overflow-y: auto; background-color: #f9f9f9;">
                    <p>点击"开始Traceroute"查看结果</p>
                </div>
            </div>
            <div class="form-actions">
                <button class="btn-save" onclick="executeTraceroute()">开始Traceroute</button>
                <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">取消</button>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
}

// 执行Traceroute
function executeTraceroute() {
    console.log('executeTraceroute called');
    const sourceId = document.getElementById('traceroute-source').value;
    const destination = document.getElementById('traceroute-destination').value;
    const resultDiv = document.getElementById('traceroute-result');
    
    // 清空之前的结果
    resultDiv.innerHTML = '';
    
    if (!destination) {
        resultDiv.innerHTML = '<p style="color: red;">错误：请输入目标IP地址</p>';
        return;
    }
    
    const source = devices.find(device => device.id === sourceId);
    if (!source) {
        resultDiv.innerHTML = '<p style="color: red;">错误：未找到源设备</p>';
        return;
    }
    
    // 显示开始信息
    const startP = document.createElement('p');
    startP.innerHTML = `开始Traceroute: <strong>${source.name} → ${destination}</strong>`;
    resultDiv.appendChild(startP);
    
    // 执行真实的Traceroute
    const hops = performTraceroute(source, destination);
    
    if (hops.length === 0) {
        const errorP = document.createElement('p');
        errorP.style.color = 'red';
        errorP.textContent = '无法到达目标主机';
        resultDiv.appendChild(errorP);
        return;
    }
    
    // 显示Traceroute结果
    hops.forEach((hop, index) => {
        setTimeout(() => {
            const hopP = document.createElement('p');
            
            // 根据不同情况显示不同的结果
            if (hop.ip === "*") {
                hopP.textContent = `${index + 1}. ${hop.ip} ${hop.time}`;
                hopP.style.color = 'red';
            } else {
                hopP.textContent = `${index + 1}. ${hop.ip} 时间=${hop.time}ms`;
            }
            
            resultDiv.appendChild(hopP);
            resultDiv.scrollTop = resultDiv.scrollHeight;
        }, (index + 1) * 1000);
    });
    
    // 显示完成信息
    setTimeout(() => {
        const endP = document.createElement('p');
        endP.innerHTML = '<strong>Traceroute完成</strong>';
        endP.style.marginTop = '10px';
        endP.style.color = 'green';
        resultDiv.appendChild(endP);
        resultDiv.scrollTop = resultDiv.scrollHeight;
    }, (hops.length + 1) * 1000);
}

// 显示NSLookup工具
function showNSLookupTool() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>域名查询测试工具</h3>
        </div>
        <div class="property-form">
            <div class="form-group">
                <label for="nslookup-source">源设备</label>
                <select id="nslookup-source">
                    ${devices.map(device => `<option value="${device.id}">${device.name}</option>`).join('')}
                </select>
            </div>
            <div class="form-group">
                <label for="nslookup-domain">域名</label>
                <input type="text" id="nslookup-domain" placeholder="www.example.com">
            </div>
            <div class="form-group">
                <label>查询类型</label>
                <select id="nslookup-record-type">
                    <option value="A">A (IPv4地址)</option>
                    <option value="AAAA">AAAA (IPv6地址)</option>
                    <option value="CNAME">CNAME (别名)</option>
                    <option value="MX">MX (邮件服务器)</option>
                    <option value="NS">NS (名称服务器)</option>
                </select>
            </div>
            <div class="form-group">
                <label>结果输出</label>
                <div id="nslookup-result" style="border: 1px solid #ccc; padding: 10px; height: 200px; overflow-y: auto; background-color: #f9f9f9;">
                    <p>点击"开始查询"查看结果</p>
                </div>
            </div>
            <div class="form-actions">
                <button class="btn-save" onclick="executeNSLookup()">开始查询</button>
                <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">取消</button>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
}

// 执行NSLookup
function executeNSLookup() {
    const sourceId = document.getElementById('nslookup-source').value;
    const domain = document.getElementById('nslookup-domain').value;
    const recordType = document.getElementById('nslookup-record-type').value;
    const resultDiv = document.getElementById('nslookup-result');
    
    // 清空之前的结果
    resultDiv.innerHTML = '';
    
    if (!domain) {
        resultDiv.innerHTML = '<p style="color: red;">错误：请输入域名</p>';
        return;
    }
    
    const source = devices.find(device => device.id === sourceId);
    if (!source) {
        resultDiv.innerHTML = '<p style="color: red;">错误：未找到源设备</p>';
        return;
    }
    
    // 显示开始信息
    const startP = document.createElement('p');
    startP.innerHTML = `开始NSLookup: <strong>${source.name} → ${domain}</strong> (${recordType})`;
    resultDiv.appendChild(startP);
    
    // 模拟DNS查询延迟
    setTimeout(() => {
        // 执行DNS查询仿真
        const dnsResult = performNSLookup(source, domain, recordType);
        
        if (dnsResult.success) {
            // 显示成功结果
            const infoP = document.createElement('p');
            infoP.innerHTML = `<strong>服务器:</strong> ${dnsResult.serverIp}`;
            resultDiv.appendChild(infoP);
            
            dnsResult.records.forEach(record => {
                const recordP = document.createElement('p');
                recordP.textContent = `${record.name}     ${record.type}     ${record.value}`;
                resultDiv.appendChild(recordP);
            });
            
            const endP = document.createElement('p');
            endP.innerHTML = `<strong>查询完成</strong>`;
            endP.style.marginTop = '10px';
            endP.style.color = 'green';
            resultDiv.appendChild(endP);
        } else {
            // 显示失败结果
            const errorP = document.createElement('p');
            errorP.style.color = 'red';
            errorP.textContent = `查询失败: ${dnsResult.error}`;
            resultDiv.appendChild(errorP);
        }
        
        resultDiv.scrollTop = resultDiv.scrollHeight;
    }, 1000);
}

// 执行NSLookup的DNS查询仿真
function performNSLookup(source, domain, recordType) {
    // 获取所有DNS服务器
    let candidateDnsServers = devices.filter(device => device.type === 'dns-server');
    
    if (candidateDnsServers.length === 0) {
        return {
            success: false,
            error: '未找到可用的DNS服务器'
        };
    }
    
    // 首先尝试使用源设备配置的DNS服务器
    const sourceDns = source.properties.dns;
    let dnsServers = [];
    
    if (sourceDns) {
        // 查找配置的DNS服务器设备
        const configuredDnsServer = candidateDnsServers.find(device => 
            getDeviceIp(device) === sourceDns
        );
        if (configuredDnsServer) {
            dnsServers.push(configuredDnsServer);
        }
    }
    
    // 如果没有配置DNS服务器或配置的DNS服务器不可用，使用所有可用的DNS服务器
    if (dnsServers.length === 0) {
        dnsServers = candidateDnsServers;
    }
    
    // 检查网络连接性：只保留源设备可以到达的DNS服务器
    const reachableDnsServers = dnsServers.filter(dnsServer => {
        const dnsServerIp = getDeviceIp(dnsServer);
        if (!dnsServerIp) {
            return false; // DNS服务器没有IP地址
        }
        return isReachable(source, dnsServerIp);
    });
    
    if (reachableDnsServers.length === 0) {
        return {
            success: false,
            error: '无法连接到任何DNS服务器，请检查网络连接'
        };
    }
    
    // 从可达的DNS服务器中选择一个
    const dnsServer = reachableDnsServers[Math.floor(Math.random() * reachableDnsServers.length)];
    
    // 从DNS服务器中查找匹配的域名记录
    const records = [];
    const matchingDomains = dnsServer.properties.domains.filter(item => 
        item.name === domain && (recordType === 'ANY' || item.type === recordType)
    );
    
    // 如果找到匹配的记录，返回实际的记录
    if (matchingDomains.length > 0) {
        matchingDomains.forEach(domainRecord => {
            let recordValue = domainRecord.ip;
            
            // 对于MX记录，返回邮件服务器的域名
            if (recordType === 'MX') {
                recordValue = `mail.${domainRecord.name}`;
            }
            
            records.push({
                name: domainRecord.name,
                type: domainRecord.type,
                value: recordValue
            });
        });
    } else {
        // 如果没有找到匹配的记录，返回错误信息
        return {
            success: false,
            error: `未找到域名 ${domain} 的 ${recordType} 记录`
        };
    }
    
    return {
        success: true,
        serverIp: dnsServer.properties.ip,
        records: records
    };
}

// 执行Traceroute
function performTraceroute(sourceDevice, targetIp) {
    const hops = [];
    const visited = new Set();
    
    // 查找目标IP所在的设备
    const targetInfo = findDeviceByIp(targetIp);
    if (!targetInfo) {
        return hops;
    }
    
    // 查找从源设备到目标设备的路径
    const path = findPath(sourceDevice, targetInfo.device);
    if (!path) {
        return hops;
    }
    
    // 检查路径中的每个路由器是否有到目标的路由
    // 与ping保持一致，严格检查路由表
    const routers = path.filter(device => device.type === 'router' || device.type === 'wireless-router');
    for (const router of routers) {
        const matchingRoute = findMatchingRoute(router, targetIp);
        if (!matchingRoute) {
            return hops; // 路由器没有到目标的路由
        }
    }
    
    // 如果是终端设备，检查是否有网关或直接连接到路由器（包括无线连接）
    if (sourceDevice.type !== 'router' && sourceDevice.type !== 'wireless-router') {
        const sourceGateway = getDeviceGateway(sourceDevice);
        let hasRouterConnection = false;
        
        // 检查是否直接连接到路由器或通过交换机连接到路由器
        for (const connection of sourceDevice.connections) {
            const neighbor = connection.device1 === sourceDevice ? connection.device2 : connection.device1;
            if (neighbor.type === 'router' || neighbor.type === 'wireless-router') {
                hasRouterConnection = true;
                break;
            }
            // 如果邻居是交换机，检查交换机是否连接到路由器
            else if (neighbor.type === 'switch') {
                for (const switchConnection of neighbor.connections) {
                    const switchNeighbor = switchConnection.device1 === neighbor ? switchConnection.device2 : switchConnection.device1;
                    if (switchNeighbor.type === 'router' || switchNeighbor.type === 'wireless-router') {
                        hasRouterConnection = true;
                        break;
                    }
                }
                if (hasRouterConnection) break;
            }
        }
        
        // 对于笔记本设备，检查是否有匹配的无线连接
        if (sourceDevice.type === 'laptop') {
            const wirelessRouter = devices.find(router => 
                router.type === 'wireless-router' && 
                router.properties.ssid === sourceDevice.properties.ssid
            );
            if (wirelessRouter) {
                hasRouterConnection = true;
            }
        }
        
        // 处理NAT场景：内网设备访问外网
        let natAllowed = false;
        if (sourceGateway) {
            // 查找网关对应的路由器
            const gatewayRouter = devices.find(device => 
                (device.type === 'router' || device.type === 'wireless-router') && 
                device.properties.interfaces.some(iface => iface.ip === sourceGateway)
            );
            
            if (gatewayRouter && gatewayRouter.properties.natEnabled) {
                // 检查源设备是否在内网
                const sourceIp = getDeviceIp(sourceDevice);
                const sourceMask = getDeviceSubnetMask(sourceDevice);
                const targetIp = targetInfo.device.properties.ip || targetInfo.interface.ip;
                
                if (sourceIp && sourceMask && targetIp) {
                    // 如果目标不在源设备的子网内，但路由器有NAT，则允许通信
                    if (!isSameSubnet(sourceIp, sourceMask, targetIp, sourceMask)) {
                        const routerRoute = findMatchingRoute(gatewayRouter, targetIp);
                        if (routerRoute) {
                            natAllowed = true;
                        }
                    }
                }
            }
        }
        
        // 如果有网关，或者有路由器连接（直接或通过交换机），则可以通信
        // 注意：即使有网关，也需要确保有物理连接到路由器
        if (!hasRouterConnection && !natAllowed) {
            // 如果没有网关，检查是否在同一子网内
            if (!sourceGateway) {
                const sourceIp = getDeviceIp(sourceDevice);
                const sourceMask = getDeviceSubnetMask(sourceDevice);
                const targetDeviceIp = getDeviceIp(targetInfo.device);
                const targetDeviceMask = getDeviceSubnetMask(targetInfo.device);
                
                if (sourceIp && sourceMask && targetDeviceIp && targetDeviceMask) {
                    if (!isSameSubnet(sourceIp, sourceMask, targetDeviceIp, targetDeviceMask)) {
                        return hops; // 不在同一子网，无法通信
                    }
                } else {
                    return hops;
                }
            } else {
                // 如果有网关，但没有物理连接到路由器，无法通信
                return hops;
            }
        }
    }
    
    // 遍历路径，收集每一跳的信息
    for (let i = 1; i < path.length; i++) {
        const device = path[i];
        
        // 获取当前跳的IP地址
        let currentIp = "";
        
        if (device.type === 'router' || device.type === 'wireless-router') {
            // 对于路由器，获取所有接口的IP
            const interfaces = device.properties.interfaces || [];
            if (interfaces.length > 0) {
                // 尝试找到与前一个设备在同一子网的接口
                const prevDevice = path[i - 1];
                
                for (const iface of interfaces) {
                    if (iface.ip && iface.subnetMask) {
                        // 检查前一个设备是否与当前接口在同一子网
                        if (prevDevice.type === 'router' || prevDevice.type === 'wireless-router') {
                            // 前一个也是路由器，检查它的所有接口
                            for (const prevIface of prevDevice.properties.interfaces || []) {
                                if (prevIface.ip && prevIface.subnetMask) {
                                    if (isSameSubnet(iface.ip, iface.subnetMask, prevIface.ip, prevIface.subnetMask)) {
                                        currentIp = iface.ip;
                                        break;
                                    }
                                }
                            }
                        } else {
                            // 前一个是终端设备
                            const prevIp = prevDevice.properties.ip || "";
                            const prevMask = prevDevice.properties.subnetMask || "";
                            if (prevIp && prevMask) {
                                if (isSameSubnet(iface.ip, iface.subnetMask, prevIp, prevMask)) {
                                    currentIp = iface.ip;
                                    break;
                                }
                            }
                        }
                    }
                    if (currentIp) break;
                }
                
                // 如果没有找到匹配的接口，使用第一个有IP的接口
                if (!currentIp) {
                    for (const iface of interfaces) {
                        if (iface.ip) {
                            currentIp = iface.ip;
                            break;
                        }
                    }
                }
            }
        } else {
            // 其他设备使用IP属性
            currentIp = device.properties.ip || "";
        }
        
        // 跳过重复的IP
        if (currentIp && !visited.has(currentIp)) {
            // 添加仿真效果
            const timeoutProbability = 0.1; // 10%的概率超时
            const lossProbability = 0.05; // 5%的概率丢包
            
            // 模拟超时
            if (Math.random() < timeoutProbability) {
                hops.push({
                    ip: "*",
                    time: "超时"
                });
            } 
            // 模拟丢包
            else if (Math.random() < lossProbability) {
                hops.push({
                    ip: "*",
                    time: "丢包"
                });
            } 
            // 正常情况
            else {
                // 根据跳数模拟更真实的延迟
                const baseDelay = 10; // 基础延迟
                const perHopDelay = 15; // 每跳增加的延迟
                const variance = 20; // 延迟变化范围
                const time = Math.floor(baseDelay + (i * perHopDelay) + (Math.random() * variance - variance/2));
                
                visited.add(currentIp);
                hops.push({
                    ip: currentIp,
                    time: time
                });
            }
        }
    }
    
    // 确保目标IP在结果中
    if (targetIp && !visited.has(targetIp)) {
        hops.push({
            ip: targetIp,
            time: Math.floor(Math.random() * 50) + 1
        });
    }
    
    return hops;
}

// 显示IP配置工具
function showIPConfigTool() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>IP 配置工具</h3>
        </div>
        <div class="property-form">
            <div class="form-group">
                <label for="ipconfig-device">选择设备</label>
                <select id="ipconfig-device">
                    ${devices.map(device => `<option value="${device.id}">${device.name}</option>`).join('')}
                </select>
            </div>
            <div class="form-actions">
                <button class="btn-save" onclick="executeIPConfig()">查看IP配置</button>
                <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">取消</button>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
}

// 显示网页浏览工具
function showWebBrowser() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    // 获取所有可用的计算机、笔记本和Web服务器
    const sourceDevices = devices.filter(device => 
        device.type === 'computer' || 
        device.type === 'laptop' || 
        device.type === 'web-server'
    );
    const webServers = devices.filter(device => device.type === 'web-server');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>网页浏览测试工具</h3>
        </div>
        <div class="property-form">
            <div class="form-group">
                <label for="browser-source">源设备</label>
                <select id="browser-source">
                    ${sourceDevices.map(device => `<option value="${device.id}">${device.name}</option>`).join('')}
                </select>
            </div>
            <div class="form-group">
                <label for="browser-destination">目标</label>
                <div style="display: flex; gap: 10px;">
                    <select id="browser-destination-type" onchange="toggleBrowserDestination()">
                        <option value="server">Web服务器</option>
                        <option value="url">URL地址</option>
                    </select>
                    <select id="browser-destination-server" style="flex: 1;">
                        ${webServers.map(server => `<option value="${server.id}">${server.name} (${server.properties.ip})</option>`).join('')}
                    </select>
                    <input type="text" id="browser-destination-url" style="flex: 1; display: none;" placeholder="http://example.com">
                </div>
            </div>
            <div class="form-actions">
                <button class="btn-save" onclick="executeWebBrowser()">开始浏览</button>
                <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">取消</button>
            </div>
            
            <!-- 浏览结果显示区域 -->
            <div class="browser-results" style="margin-top: 10px; padding: 15px; background-color: #f8f9fa; border-radius: 5px;">
                <div id="browser-output" style="color: #2c3e50;"></div>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
}

// 切换网页浏览目标类型
function toggleBrowserDestination() {
    const type = document.getElementById('browser-destination-type').value;
    const serverSelect = document.getElementById('browser-destination-server');
    const urlInput = document.getElementById('browser-destination-url');
    
    if (type === 'server') {
        serverSelect.style.display = 'block';
        urlInput.style.display = 'none';
    } else {
        serverSelect.style.display = 'none';
        urlInput.style.display = 'block';
    }
}

// 执行网页浏览
function executeWebBrowser() {
    const sourceId = document.getElementById('browser-source').value;
    const destinationType = document.getElementById('browser-destination-type').value;
    
    let destination = null;
    let destinationName = null;
    
    if (destinationType === 'server') {
        const serverId = document.getElementById('browser-destination-server').value;
        destination = devices.find(device => device.id === serverId);
        destinationName = destination.name;
    } else {
        const url = document.getElementById('browser-destination-url').value;
        destination = { type: 'url', address: url };
        destinationName = url;
    }
    
    const source = devices.find(device => device.id === sourceId);
    if (source && destination) {
        const browserOutput = document.getElementById('browser-output');
        browserOutput.innerHTML = '';
        
        // 记录日志
        log(`开始网页浏览: ${source.name} → ${destinationName}`);
        
        // 模拟DNS解析过程
        browserOutput.innerHTML += `<div style="color: #3498db; font-weight: bold;">正在解析目标地址...</div>`;
        
        setTimeout(() => {
            // 首先检查源设备是否有有效的IP地址
            const sourceIp = getDeviceIp(source);
            if (!sourceIp) {
                browserOutput.innerHTML += `<div style="color: #e74c3c;">✗ 错误: 源设备 ${source.name} 没有有效的IP地址</div>`;
                if (source.type === 'laptop' && source.properties.ipMode === 'dhcp') {
                    browserOutput.innerHTML += `<div style="color: #e74c3c; margin-top: 5px;">提示: 设备使用DHCP模式但尚未连接到WiFi路由器，无法获取IP地址</div>`;
                }
                log(`网页浏览失败: ${source.name} 没有有效的IP地址`);
                return;
            }
            
            let targetIP = null;
            let dnsResult = null;
            
            if (destination.type === 'web-server') {
                // 使用统一的IP获取函数，确保正确处理所有设备类型
                targetIP = getDeviceIp(destination);
                if (!targetIP) {
                    browserOutput.innerHTML += `<div style="color: #e74c3c;">✗ 错误: 目标Web服务器没有有效的IP地址</div>`;
                    log(`网页浏览失败: 目标Web服务器 ${destination.name} 没有有效的IP地址`);
                    return;
                }
                dnsResult = `<div style="color: #27ae60;">✓ 直接连接到Web服务器: ${targetIP}</div>`;
            } else {
                // 模拟DNS解析
                const dnsServers = devices.filter(device => device.type === 'dns-server');
                let resolved = false;
                
                for (const dnsServer of dnsServers) {
                    for (const domain of dnsServer.properties.domains) {
                        if (destination.address.includes(domain.name)) {
                            targetIP = domain.ip;
                            resolved = true;
                            break;
                        }
                    }
                    if (resolved) break;
                }
                
                if (resolved) {
                    dnsResult = `<div style="color: #27ae60;">✓ DNS解析成功: ${destination.address} → ${targetIP}</div>`;
                } else {
                    dnsResult = `<div style="color: #e74c3c;">✗ DNS解析失败: 无法找到 ${destination.address}</div>`;
                }
            }
            
            browserOutput.innerHTML += dnsResult;
            
            if (targetIP) {
                // 检查网络可达性
                const isNetworkReachable = isReachable(source, targetIP);
                
                if (!isNetworkReachable) {
                    browserOutput.innerHTML += `<div style="color: #e74c3c;">✗ 网络不可达: 源设备 ${source.name} 无法到达目标 ${targetIP}</div>`;
                    browserOutput.innerHTML += `<div style="color: #e74c3c; margin-top: 5px;">可能原因：设备未连接、不在同一网络或路由配置不正确</div>`;
                    log(`网页浏览失败: ${source.name} 无法到达 ${targetIP} (网络不可达)`);
                    return;
                }
                
                // 模拟TCP连接建立
                browserOutput.innerHTML += `<div style="color: #3498db;">正在建立TCP连接...</div>`;
                
                setTimeout(() => {
                    // 再次检查网络可达性（防止在延迟期间网络状态改变）
                    const stillReachable = isReachable(source, targetIP);
                    if (!stillReachable) {
                        browserOutput.innerHTML += `<div style="color: #e74c3c;">✗ TCP连接失败: 网络不可达</div>`;
                        log(`网页浏览失败: ${source.name} 无法建立TCP连接到 ${targetIP}`);
                        return;
                    }
                    
                    // 模拟HTTP请求发送
                    browserOutput.innerHTML += `<div style="color: #27ae60;">✓ TCP连接已建立</div>`;
                    browserOutput.innerHTML += `<div style="color: #3498db;">正在发送HTTP请求...</div>`;
                    
                    setTimeout(() => {
                        // 查找对应的Web服务器
                        // 如果destination已经是web-server设备对象，直接使用它；否则通过IP查找
                        let targetServer = null;
                        if (destination && destination.type === 'web-server') {
                            // 直接使用已选择的服务器对象，确保使用最新的内容
                            targetServer = destination;
                        } else {
                            // 通过IP查找服务器（用于URL浏览的情况）
                            targetServer = devices.find(server => server.type === 'web-server' && server.properties.ip === targetIP);
                        }
                        
                        if (targetServer) {
                            // 最终检查：确保目标服务器仍然可达
                            const finalCheck = isReachable(source, targetIP);
                            if (!finalCheck) {
                                browserOutput.innerHTML += `<div style="color: #e74c3c;">✗ HTTP请求失败: 网络连接中断</div>`;
                                log(`网页浏览失败: ${source.name} 无法访问 ${destinationName} (网络连接中断)`);
                                return;
                            }
                            
                            // 模拟HTTP响应
                            browserOutput.innerHTML += `<div style="color: #27ae60;">✓ 收到HTTP响应 (200 OK)</div>`;
                            browserOutput.innerHTML += `<div style="margin-top: 10px; padding: 10px; background-color: white; border-radius: 4px; border-left: 4px solid #3498db;">`;
                            browserOutput.innerHTML += `<h4 style="margin: 0 0 10px 0; color: #2c3e50;">${targetServer.properties.website.title}</h4>`;
                            browserOutput.innerHTML += `<p style="margin: 0; color: #555;">${targetServer.properties.website.content}</p>`;
                            browserOutput.innerHTML += `</div>`;
                            
                            // 记录日志
                            log(`网页浏览成功: ${source.name} 成功访问了 ${destinationName}`);
                        } else {
                            browserOutput.innerHTML += `<div style="color: #e74c3c;">✗ 无法连接到目标服务器</div>`;
                            log(`网页浏览失败: ${source.name} 无法连接到 ${destinationName}`);
                        }
                    }, 800);
                }, 800);
            } else {
                log(`网页浏览失败: ${source.name} 无法解析 ${destinationName}`);
            }
        }, 800);
    }
}

// 执行IP配置查看 (已被网页浏览工具替换)
function showIPConfigTool() {
    alert('IP配置工具已被网页浏览工具替换');
}

// 执行IP配置查看
function executeIPConfig() {
    // 已被网页浏览工具替换
}

// 显示帮助
function showHelp() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>MixNET 网络仿真软件使用帮助</h3>
        </div>
        <div style="max-height: 600px; overflow-y: auto; padding: 20px; line-height: 1.6;">
            <!-- 软件界面概览 -->
            <h4 style="color: #3498db; margin-top: 0;">软件界面概览</h4>
            <p style="margin-bottom: 20px;">软件界面分为四个主要部分：</p>
            <ul style="margin-bottom: 20px;">
                <li><strong>设备库（左侧）：</strong>提供各种网络设备，点击即可添加到画布</li>
                <li><strong>画布区域（中央）：</strong>用于构建和显示网络拓扑，可拖拽移动设备</li>
                <li><strong>属性面板（右侧）：</strong>显示选中设备的详细信息和配置选项</li>
                <li><strong>工具栏（顶部）：</strong>提供拓扑管理、设备连接和仿真控制功能</li>
            </ul>
            
            <!-- 一、快速开始 -->
            <h4 style="color: #3498db;">一、快速开始</h4>
            
            <h5 style="color: #2c3e50; margin-bottom: 10px;">1. 添加设备</h5>
            <div style="margin-bottom: 20px;">
                <ol style="padding-left: 20px;">
                    <li>从左侧设备库中选择要添加的设备图标（如台式机、路由器等）</li>
                    <li>在中央画布上点击要放置设备的位置</li>
                    <li>设备将自动出现在画布上，可通过拖拽调整位置</li>
                </ol>
            </div>
            
            <h5 style="color: #2c3e50; margin-bottom: 10px;">2. 连接设备</h5>
            <div style="margin-bottom: 20px;">
                <ol style="padding-left: 20px;">
                    <li>点击顶部工具栏的"连接设备"按钮</li>
                    <li>依次点击要连接的两个设备（注意连接顺序不影响结果）</li>
                    <li>设备之间会出现连接线表示物理连接</li>
                    <li>再次点击"连接设备"按钮可退出连接模式</li>
                </ol>
            </div>
            
            <h5 style="color: #2c3e50; margin-bottom: 10px;">3. 配置设备</h5>
            <div style="margin-bottom: 20px;">
                <ol style="padding-left: 20px;">
                    <li>点击设备图标选中它（设备会高亮显示）</li>
                    <li>右侧属性面板会显示该设备的详细信息和配置选项</li>
                    <li>根据需要修改配置参数（如IP地址、子网掩码、默认网关等）</li>
                    <li>修改完成后，点击"保存配置"按钮保存更改</li>
                    <li>设备图标下方会显示关键配置信息，方便查看</li>
                </ol>
            </div>
            
            <h5 style="color: #2c3e50; margin-bottom: 10px;">4. 运行仿真</h5>
            <div style="margin-bottom: 20px;">
                <ol style="padding-left: 20px;">
                    <li>点击顶部工具栏的"运行仿真"按钮（按钮会变为"停止仿真"）</li>
                    <li>仿真启动后，各实验工具按钮变为可用状态</li>
                    <li>此时可以使用各种实验工具（如Ping测试、网络命令等）测试网络连接和功能</li>
                    <li>完成测试后，点击"停止仿真"按钮结束仿真</li>
                </ol>
            </div>
            
            <!-- 二、设备类型详解 -->
            <h4 style="color: #3498db; margin-top: 30px;">二、设备类型详解</h4>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-bottom: 20px;">
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px; border: 1px solid #e9ecef;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">台式机/笔记本</h6>
                    <p style="font-size: 14px; margin: 0;">终端设备，可配置IP地址、子网掩码、默认网关、DNS服务器等网络参数。笔记本支持有线和无线连接。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px; border: 1px solid #e9ecef;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">路由器/WiFi路由器</h6>
                    <p style="font-size: 14px; margin: 0;">网络层设备，用于连接不同网络。可配置多个接口IP地址和路由表。WiFi路由器支持有线和无线连接，可配置SSID和密码。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px; border: 1px solid #e9ecef;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">交换机</h6>
                    <p style="font-size: 14px; margin: 0;">数据链路层设备，用于局域网内设备间的数据转发，可配置端口数量，支持多设备同时通信。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px; border: 1px solid #e9ecef;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">服务器设备</h6>
                    <p style="font-size: 14px; margin: 0;">包括Web服务器（提供HTTP服务）、DNS服务器（域名解析）和Email服务器（邮件收发），提供各种网络服务。</p>
                </div>
            </div>
            
            <!-- 三、实验工具使用 -->
            <h4 style="color: #3498db; margin-top: 30px;">三、实验工具使用</h4>
            
            <h5 style="color: #2c3e50; margin-bottom: 15px;">网络命令测试工具</h5>
            <div style="background-color: #f8f9fa; padding: 20px; border-radius: 5px; margin-bottom: 20px;">
                <p>支持多种网络命令测试，包括：</p>
                <div style="display: flex; flex-wrap: wrap; gap: 15px;">
                    <div>
                        <strong>Windows命令：</strong>arp, ipconfig, ping, tracert, nslookup, route
                    </div>
                    <div>
                        <strong>Linux命令：</strong>arp, ifconfig, ping, traceroute, nslookup, route
                    </div>
                </div>
                <p style="margin-top: 10px;">使用说明：</p>
                <ol style="padding-left: 20px;">
                    <li>选择源设备和操作系统</li>
                    <li>在终端中输入命令（支持上下箭头浏览历史）</li>
                    <li>按Enter执行命令，查看输出结果</li>
                    <li>按Ctrl+L清屏</li>
                    <li>nslookup命令支持交互式模式</li>
                </ol>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-bottom: 20px;">
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">Ping测试</h6>
                    <p style="font-size: 14px;">测试网络连通性，发送ICMP数据包检测目标设备是否可达，显示响应时间和丢包率。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">Traceroute</h6>
                    <p style="font-size: 14px;">追踪数据包从源设备到目标设备的路径，显示经过的路由器和每跳的延迟时间，帮助诊断网络问题。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">网页浏览</h6>
                    <p style="font-size: 14px;">模拟浏览器访问Web服务器，输入URL即可查看网页内容，支持域名解析。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">邮件测试</h6>
                    <p style="font-size: 14px;">发送和接收邮件，测试邮件服务器的SMTP（发送）和POP3/IMAP（接收）功能。</p>
                </div>
            </div>
            
            <!-- 四、复杂路由配置预设实验 -->
            <h4 style="color: #3498db; margin-top: 30px;">四、复杂路由配置预设实验</h4>
            <div style="margin-bottom: 20px;">
                <p>本实验预设了一个复杂的路由网络拓扑，包含多个路由器和不同子网：</p>
                <ul style="padding-left: 20px; margin-bottom: 15px;">
                    <li>3个路由器连接4个子网</li>
                    <li>配置了静态路由和默认路由</li>
                    <li>包含终端设备和服务器</li>
                    <li>支持跨子网通信测试</li>
                </ul>
                <p>使用方法：</p>
                <ol style="padding-left: 20px;">
                    <li>从预设实验下拉菜单中选择"复杂路由配置"</li>
                    <li>系统自动加载完整的网络拓扑结构</li>
                    <li>点击"运行仿真"按钮启动仿真</li>
                    <li>使用Ping测试、Traceroute等工具测试各设备间的连通性</li>
                    <li>点击路由器查看路由表配置，了解数据包转发路径</li>
                </ol>
            </div>
            
            <!-- 五、拓扑管理 -->
            <h4 style="color: #3498db; margin-top: 30px;">五、拓扑管理</h4>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-bottom: 20px;">
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">新建拓扑</h6>
                    <p>清空当前画布，创建新的网络拓扑（会提示确认）。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">保存拓扑</h6>
                    <p>将当前网络拓扑保存为JSON文件，包含所有设备配置和连接关系。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">打开拓扑</h6>
                    <p>从JSON文件加载之前保存的网络拓扑，快速恢复实验环境。</p>
                </div>
                
                <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px;">
                    <h6 style="color: #2c3e50; margin: 0 0 10px 0;">运行仿真</h6>
                    <p>启动网络仿真模式，测试网络连接和进行各种网络实验。</p>
                </div>
            </div>
            
            <!-- 六、常见问题与解决方案 -->
            <h4 style="color: #3498db; margin-top: 30px;">六、常见问题与解决方案</h4>
            <div style="background-color: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; border-radius: 4px; margin-bottom: 20px;">
                <h6 style="color: #856404; margin: 0 0 10px 0;">设备无法通信？</h6>
                <ul style="margin: 0; padding-left: 20px;">
                    <li>检查IP地址和子网掩码是否配置正确</li>
                    <li>确认设备之间有物理连接</li>
                    <li>检查路由器路由表配置</li>
                    <li>确保默认网关设置正确</li>
                </ul>
            </div>
            
            <div style="background-color: #d1ecf1; border-left: 4px solid #17a2b8; padding: 15px; border-radius: 4px; margin-bottom: 20px;">
                <h6 style="color: #0c5460; margin: 0 0 10px 0;">DNS解析失败？</h6>
                <ul style="margin: 0; padding-left: 20px;">
                    <li>检查DNS服务器配置</li>
                    <li>确认DNS记录（A记录、MX记录）是否正确</li>
                    <li>测试DNS服务器连通性</li>
                </ul>
            </div>
            
            <div style="background-color: #f8d7da; border-left: 4px solid #dc3545; padding: 15px; border-radius: 4px; margin-bottom: 20px;">
                <h6 style="color: #721c24; margin: 0 0 10px 0;">仿真工具不可用？</h6>
                <ul style="margin: 0; padding-left: 20px;">
                    <li>确保已点击"运行仿真"按钮</li>
                    <li>检查设备配置是否完整</li>
                    <li>确认网络拓扑连接正确</li>
                </ul>
            </div>
            
            <!-- 七、使用技巧 -->
            <h4 style="color: #3498db; margin-top: 30px;">七、使用技巧</h4>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 15px; margin-bottom: 20px;">
                <div style="background-color: #f8f9fa; padding: 12px; border-radius: 5px; font-size: 14px;">
                    <strong>设备信息查看：</strong>设备图标下方会显示关键网络信息
                </div>
                <div style="background-color: #f8f9fa; padding: 12px; border-radius: 5px; font-size: 14px;">
                    <strong>快速配置：</strong>使用预设实验快速搭建实验环境
                </div>
                <div style="background-color: #f8f9fa; padding: 12px; border-radius: 5px; font-size: 14px;">
                    <strong>命令历史：</strong>使用上下箭头快速浏览和重复执行命令
                </div>
                <div style="background-color: #f8f9fa; padding: 12px; border-radius: 5px; font-size: 14px;">
                    <strong>拓扑保存：</strong>定期保存拓扑配置，避免数据丢失
                </div>
            </div>
            
            <!-- 结尾 -->
            <div style="background-color: #ecf0f1; padding: 20px; border-radius: 5px; margin-top: 30px;">
                <h5 style="color: #2c3e50; margin: 0 0 10px 0;">开始您的网络实验之旅！</h5>
                <p style="margin: 0;">MixNET网络仿真软件提供了丰富的网络设备和实验工具，帮助您学习和理解网络原理。通过实际操作和测试，您将更好地掌握网络配置和故障排除技能。</p>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
}

// 网络工具函数
// 将IP地址转换为整数
function ipToInt(ip) {
    if (!ip) return 0;
    return ip.split('.').reduce((acc, octet) => (acc << 8) + parseInt(octet, 10), 0) >>> 0;
}

// 将子网掩码转换为整数
function maskToInt(mask) {
    if (!mask) return 0;
    return ipToInt(mask);
}

// 计算网络地址
function getNetworkAddress(ip, mask) {
    const ipInt = ipToInt(ip);
    const maskInt = maskToInt(mask);
    return (ipInt & maskInt) >>> 0;
}

// 判断两个IP是否在同一子网
function isSameSubnet(ip1, mask1, ip2, mask2) {
    // 如果任何一个IP或掩码为null，返回false
    if (!ip1 || !mask1 || !ip2 || !mask2) {
        return false;
    }
    // 两个设备必须在同一个物理网络才能直接通信
    const network1 = getNetworkAddress(ip1, mask1);
    const network2 = getNetworkAddress(ip2, mask2);
    return network1 === network2;
}

// 获取设备的IP地址（考虑笔记本的DHCP模式）
function getDeviceIp(device) {
    if (device.type === 'router' || device.type === 'wireless-router') {
        return device.properties.interfaces.length > 0 ? device.properties.interfaces[0].ip : null;
    } else if (device.type === 'laptop' && device.properties.ipMode === 'dhcp') {
        // DHCP模式：检查是否有匹配的路由器并已分配IP
        const wirelessRouter = devices.find(router => 
            router.type === 'wireless-router' && 
            router.properties.ssid === device.properties.ssid
        );
        
        if (wirelessRouter && device.properties.ip) {
            // 有匹配的路由器且已分配IP，返回DHCP分配的IP
            return device.properties.ip;
        } else {
            // 没有匹配的路由器或未分配IP，返回null（表示未配置）
            return null;
        }
    } else {
        // 静态模式或其他设备类型：直接返回配置的IP
        return device.properties.ip || null;
    }
}

// 获取设备的子网掩码
function getDeviceSubnetMask(device) {
    if (device.type === 'router' || device.type === 'wireless-router') {
        return device.properties.interfaces.length > 0 ? device.properties.interfaces[0].subnetMask : null;
    } else {
        return device.properties.subnetMask || null;
    }
}

// 获取设备的默认网关
function getDeviceGateway(device) {
    if (device.type !== 'router') {
        return device.properties.gateway || null;
    }
    return null; // 路由器不需要网关
}

// 查找具有特定IP地址的设备
function findDeviceByIp(targetIp) {
    for (const device of devices) {
        if (device.type === 'router' || device.type === 'wireless-router') {
            // 检查路由器的所有接口
            for (const iface of device.properties.interfaces) {
                if (iface.ip === targetIp) {
                    return { device, interface: iface };
                }
            }
        } else {
            // 检查计算机、服务器等设备的IP
            if (device.properties.ip === targetIp) {
                return { device, interface: null };
            }
        }
    }
    return null;
}

// 计算从源设备到目标设备的路径
function findPath(sourceDevice, targetDevice) {
    // 使用广度优先搜索查找路径
    const queue = [{ device: sourceDevice, path: [sourceDevice] }];
    const visited = new Set([sourceDevice.id]);
    
    while (queue.length > 0) {
        const { device, path } = queue.shift();
        
        // 如果到达目标设备，返回路径
        if (device === targetDevice) {
            return path;
        }
        
        // 检查所有物理连接的设备
        for (const connection of device.connections) {
            const neighbor = connection.device1 === device ? connection.device2 : connection.device1;
            
            if (!visited.has(neighbor.id)) {
                visited.add(neighbor.id);
                queue.push({ device: neighbor, path: [...path, neighbor] });
            }
        }
        
        // 检查无线连接（对于笔记本设备）
        if (device.type === 'laptop' && device.properties.wirelessEnabled) {
            const wirelessRouter = devices.find(router => 
                router.type === 'wireless-router' && 
                router.properties.ssid === device.properties.ssid
            );
            
            if (wirelessRouter && !visited.has(wirelessRouter.id)) {
                // 检查密码是否正确
                let passwordMatch = true;
                if (wirelessRouter.properties.security && wirelessRouter.properties.security !== 'open') {
                    passwordMatch = (device.properties.password === wirelessRouter.properties.password);
                }
                
                if (passwordMatch) {
                    // 检查笔记本电脑和无线路由器的无线接口是否在同一个网段
                    const wirelessInterface = wirelessRouter.properties.interfaces.find(iface => iface.name.startsWith('wlan'));
                    if (wirelessInterface) {
                        const laptopIp = device.properties.ip;
                        const laptopMask = device.properties.subnetMask;
                        const routerIp = wirelessInterface.ip;
                        const routerMask = wirelessInterface.subnetMask;
                        
                        if (isSameSubnet(laptopIp, laptopMask, routerIp, routerMask)) {
                            visited.add(wirelessRouter.id);
                            queue.push({ device: wirelessRouter, path: [...path, wirelessRouter] });
                        }
                    }
                }
            }
        }
        
        // 检查无线路由器的无线连接
        if (device.type === 'wireless-router') {
            const connectedLaptops = devices.filter(laptop => 
                laptop.type === 'laptop' && 

                laptop.properties.ssid === device.properties.ssid
            );
            
            // 找到无线路由器的无线接口
            const wirelessInterface = device.properties.interfaces.find(iface => iface.name.startsWith('wlan'));
            
            for (const laptop of connectedLaptops) {
                if (!visited.has(laptop.id) && wirelessInterface) {
                    // 检查密码是否正确
                    let passwordMatch = true;
                    if (device.properties.security && device.properties.security !== 'open') {
                        passwordMatch = (laptop.properties.password === device.properties.password);
                    }
                    
                    if (passwordMatch) {
                        // 检查笔记本电脑和无线路由器的无线接口是否在同一个网段
                        const laptopIp = laptop.properties.ip;
                        const laptopMask = laptop.properties.subnetMask;
                        const routerIp = wirelessInterface.ip;
                        const routerMask = wirelessInterface.subnetMask;
                        
                        if (isSameSubnet(laptopIp, laptopMask, routerIp, routerMask)) {
                            visited.add(laptop.id);
                            queue.push({ device: laptop, path: [...path, laptop] });
                        }
                    }
                }
            }
        }
    }
    
    // 没有找到路径
    return null;
}

// 检查网络连接是否可达
// 重写的isReachable函数，更严格的路由检查
function isReachable(sourceDevice, targetIp) {
    // 查找目标IP所在的设备
    const targetInfo = findDeviceByIp(targetIp);
    if (!targetInfo) {
        return false; // 目标IP不存在
    }
    
    // 如果源设备就是目标设备，直接返回可达
    if (sourceDevice === targetInfo.device) {
        return true;
    }
    
    // 首先检查是否有物理连接路径
    const path = findPath(sourceDevice, targetInfo.device);
    if (!path) {
        return false; // 没有物理连接路径
    }
    
    // 优先检查：如果源设备和目标设备在同一子网内，检查是否可以直接通信（不经过路由器）
    const sourceIp = getDeviceIp(sourceDevice);
    const sourceMask = getDeviceSubnetMask(sourceDevice);
    const targetDeviceIp = getDeviceIp(targetInfo.device);
    const targetDeviceMask = getDeviceSubnetMask(targetInfo.device);
    
    if (sourceIp && sourceMask && targetDeviceIp && targetDeviceMask) {
        if (isSameSubnet(sourceIp, sourceMask, targetDeviceIp, targetDeviceMask)) {
            // 同子网内，检查路径中是否必须经过路由器
            const routersInPath = path.filter(device => device.type === 'router' || device.type === 'wireless-router');
            
            // 如果路径中没有路由器，或者只有交换机，可以直接通信
            if (routersInPath.length === 0) {
                return true; // 同子网且不经过路由器，直接通信
            }
            
            // 如果路径中有路由器，检查数据包是否真的需要经过路由器
            // 对于同子网设备，如果它们通过交换机连接，数据包不需要经过路由器
            // 检查路径中是否有交换机，且源设备和目标设备都连接到同一个交换机
            const switchesInPath = path.filter(device => device.type === 'switch');
            if (switchesInPath.length > 0) {
                // 检查源设备和目标设备是否都连接到同一个交换机
                for (const switchDevice of switchesInPath) {
                    const sourceConnectedToSwitch = sourceDevice.connections.some(conn => {
                        const neighbor = conn.device1 === sourceDevice ? conn.device2 : conn.device1;
                        return neighbor === switchDevice;
                    });
                    const targetConnectedToSwitch = targetInfo.device.connections.some(conn => {
                        const neighbor = conn.device1 === targetInfo.device ? conn.device2 : conn.device1;
                        return neighbor === switchDevice;
                    });
                    
                    if (sourceConnectedToSwitch && targetConnectedToSwitch) {
                        // 源设备和目标设备都连接到同一个交换机，可以直接通信，不需要经过路由器
                        return true;
                    }
                }
            }
            
            // 如果同子网但必须经过路由器，检查路由器的直连路由
            // 对于同子网设备，路由器应该有直连路由
            for (const router of routersInPath) {
                let hasDirectRoute = false;
                for (const iface of router.properties.interfaces) {
                    if (isSameSubnet(targetIp, iface.subnetMask, iface.ip, iface.subnetMask)) {
                        hasDirectRoute = true;
                        break;
                    }
                }
                if (!hasDirectRoute) {
                    // 同子网但路由器没有直连路由，这是配置错误，但允许通信
                    // 因为同子网设备应该能直接通信
                    continue;
                }
            }
            // 同子网设备，允许通信
            return true;
        }
    }
    
    // 不同子网，需要检查路由器的路由
    // 检查路径中的每个路由器是否有到目标的路由
    const routers = path.filter(device => device.type === 'router' || device.type === 'wireless-router');
    for (const router of routers) {
        // 首先检查目标是否在路由器的直连子网内
        let hasDirectRoute = false;
        for (const iface of router.properties.interfaces) {
            if (isSameSubnet(targetIp, iface.subnetMask, iface.ip, iface.subnetMask)) {
                hasDirectRoute = true;
                break;
            }
        }
        
        // 如果不在直连子网内，检查路由表
        if (!hasDirectRoute) {
            const matchingRoute = findMatchingRoute(router, targetIp);
            if (!matchingRoute) {
                return false; // 路由器没有到目标的路由
            }
            
            // 确保路由不是黑洞路由或不可达路由
            if (matchingRoute.nextHop === '0.0.0.0' || matchingRoute.nextHop === 'unreachable') {
                return false;
            }
            
            // 对于非直连路由，必须确保下一跳IP在路由器的某个接口子网内
            if (matchingRoute.nextHop) {
                let hasNextHopInterface = false;
                for (const iface of router.properties.interfaces) {
                    if (isSameSubnet(matchingRoute.nextHop, iface.subnetMask, iface.ip, iface.subnetMask)) {
                        hasNextHopInterface = true;
                        break;
                    }
                }
                if (!hasNextHopInterface) {
                    return false; // 下一跳IP不在路由器的任何接口子网内
                }
            }
        }
    }
    
    // 检查目标设备的网关配置是否正确
    // 如果目标设备是终端设备（非路由器），检查其网关是否存在于路径中某个路由器的接口上
    // 但是，如果目标在路径中某个路由器的直连子网内，即使网关配置不正确，也允许访问（直连网络不需要网关）
    if (targetInfo.device.type !== 'router' && targetInfo.device.type !== 'wireless-router') {
        // 首先检查目标是否在路径中某个路由器的直连子网内
        let isInDirectSubnet = false;
        for (const router of routers) {
            for (const iface of router.properties.interfaces) {
                const targetDeviceIp = targetInfo.device.properties.ip;
                if (targetDeviceIp && isSameSubnet(targetDeviceIp, iface.subnetMask, iface.ip, iface.subnetMask)) {
                    isInDirectSubnet = true;
                    break;
                }
            }
            if (isInDirectSubnet) break;
        }
        
        // 如果目标在直连子网内，不需要检查网关配置（直连网络不需要网关）
        if (!isInDirectSubnet) {
            const targetGateway = getDeviceGateway(targetInfo.device);
            if (targetGateway) {
                // 查找网关对应的路由器
                const gatewayRouter = devices.find(device => 
                    (device.type === 'router' || device.type === 'wireless-router') && 
                    device.properties.interfaces.some(iface => iface.ip === targetGateway)
                );
                
                // 如果找不到网关对应的路由器，说明网关配置错误
                if (!gatewayRouter) {
                    return false; // 目标设备的网关IP不存在于任何路由器接口上
                }
                
                // 检查网关路由器是否在路径中
                if (!path.includes(gatewayRouter)) {
                    return false; // 网关路由器不在路径中
                }
                
                // 检查目标设备的IP是否在网关路由器的某个接口子网内
                const targetDeviceIp = targetInfo.device.properties.ip;
                if (targetDeviceIp) {
                    let isInGatewaySubnet = false;
                    for (const iface of gatewayRouter.properties.interfaces) {
                        if (iface.ip === targetGateway && isSameSubnet(targetDeviceIp, iface.subnetMask, iface.ip, iface.subnetMask)) {
                            isInGatewaySubnet = true;
                            break;
                        }
                    }
                    if (!isInGatewaySubnet) {
                        return false; // 目标设备的IP不在其网关路由器的接口子网内
                    }
                }
            }
        }
    }
    
    // 如果是终端设备，检查是否有网关或直接连接到路由器
    if (sourceDevice.type !== 'router' && sourceDevice.type !== 'wireless-router') {
        const sourceGateway = getDeviceGateway(sourceDevice);
        let hasRouterConnection = false;
        
        // 检查是否直接连接到路由器或通过交换机连接到路由器
        for (const connection of sourceDevice.connections) {
            const neighbor = connection.device1 === sourceDevice ? connection.device2 : connection.device1;
            if (neighbor.type === 'router' || neighbor.type === 'wireless-router') {
                hasRouterConnection = true;
                break;
            }
            // 如果邻居是交换机，检查交换机是否连接到路由器
            else if (neighbor.type === 'switch') {
                for (const switchConnection of neighbor.connections) {
                    const switchNeighbor = switchConnection.device1 === neighbor ? switchConnection.device2 : switchConnection.device1;
                    if (switchNeighbor.type === 'router' || switchNeighbor.type === 'wireless-router') {
                        hasRouterConnection = true;
                        break;
                    }
                }
                if (hasRouterConnection) break;
            }
        }
        
        // 检查是否有无线连接（针对笔记本电脑）
        let hasWirelessConnection = false;
        if (!hasRouterConnection && sourceDevice.type === 'laptop' && sourceDevice.properties.ssid) {
            // 查找相同SSID的WiFi路由器
            const wirelessRouter = devices.find(device => 
                device.type === 'wireless-router' && 
                device.properties.ssid === sourceDevice.properties.ssid
            );
            
            if (wirelessRouter) {
                hasWirelessConnection = true;
            }
        }
        
        // 处理NAT场景：内网设备访问外网
        let natAllowed = false;
        if (sourceGateway) {
            // 查找网关对应的路由器
            const gatewayRouter = devices.find(device => 
                (device.type === 'router' || device.type === 'wireless-router') && 
                device.properties.interfaces.some(iface => iface.ip === sourceGateway)
            );
            
            if (gatewayRouter && gatewayRouter.properties.natEnabled) {
                // 检查源设备是否在内网
                const sourceIp = getDeviceIp(sourceDevice);
                const sourceMask = getDeviceSubnetMask(sourceDevice);
                const targetDeviceIp = targetInfo.device.properties.ip || targetInfo.interface.ip;
                
                if (sourceIp && sourceMask && targetDeviceIp) {
                    // 如果目标不在源设备的子网内，但路由器有NAT，则允许通信
                    if (!isSameSubnet(sourceIp, sourceMask, targetDeviceIp, sourceMask)) {
                        const routerRoute = findMatchingRoute(gatewayRouter, targetDeviceIp);
                        if (routerRoute) {
                            natAllowed = true;
                        }
                    }
                }
            }
        }
        
        // 如果有网关，或者有路由器连接（直接或通过交换机），或者有无线连接，或者NAT允许，则可以通信
        // 注意：即使有网关，也需要确保有物理连接到路由器或无线连接
        if (!hasRouterConnection && !hasWirelessConnection && !natAllowed) {
            // 如果没有网关，检查是否在同一子网内
            if (!sourceGateway) {
                const sourceIp = getDeviceIp(sourceDevice);
                const sourceMask = getDeviceSubnetMask(sourceDevice);
                const targetDeviceIp = getDeviceIp(targetInfo.device);
                const targetDeviceMask = getDeviceSubnetMask(targetInfo.device);
                
                if (sourceIp && sourceMask && targetDeviceIp && targetDeviceMask) {
                    if (!isSameSubnet(sourceIp, sourceMask, targetDeviceIp, targetDeviceMask)) {
                        return false; // 不在同一子网，无法通信
                    }
                } else {
                    return false;
                }
            } else {
                // 如果有网关，但没有物理连接到路由器，无法通信
                return false;
            }
        }
    }
    
    // 如果所有检查都通过，返回可达
    return true;
}

// 检查路由表中是否有匹配的路由项
function findMatchingRoute(router, targetIp) {
    // 首先检查直连路由（优先级最高）
    for (const iface of router.properties.interfaces) {
        if (isSameSubnet(targetIp, iface.subnetMask, iface.ip, iface.subnetMask)) {
            return { 
                network: getNetworkAddress(iface.ip, iface.subnetMask),
                subnetMask: iface.subnetMask,
                nextHop: "",
                interface: iface.name
            };
        }
    }
    
    // 然后检查路由表中的路由（最长前缀匹配）
    let bestMatch = null;
    let longestMask = 0;
    
    for (const route of router.properties.routingTable) {
        // 获取路由的子网掩码长度
        const maskLength = getMaskLength(route.subnetMask);
        
        // 检查目标IP是否在路由网络中
        if (isSameSubnet(targetIp, route.subnetMask, route.network, route.subnetMask)) {
            // 确保路由有有效的下一跳或直连
            if (route.network === '0.0.0.0' && route.subnetMask === '0.0.0.0') {
                // 默认路由必须有下一跳
                if (!route.nextHop) {
                    continue;
                }
            } else {
                // 非默认路由必须要么是直连（无下一跳）要么有有效的下一跳
                if (route.nextHop === '0.0.0.0' || route.nextHop === 'unreachable') {
                    continue;
                }
            }
            
            // 如果是更长的掩码匹配，更新最佳匹配
            if (maskLength > longestMask) {
                bestMatch = route;
                longestMask = maskLength;
            }
        }
    }
    
    // 如果找到匹配的路由，需要确定出接口
    if (bestMatch) {
        // 如果有下一跳IP，根据下一跳IP确定出接口
        if (bestMatch.nextHop) {
            // 查找连接到下一跳IP的接口
            for (const iface of router.properties.interfaces) {
                if (isSameSubnet(bestMatch.nextHop, iface.subnetMask, iface.ip, iface.subnetMask)) {
                    // 返回带有确定出接口的路由
                    return {
                        network: bestMatch.network,
                        subnetMask: bestMatch.subnetMask,
                        nextHop: bestMatch.nextHop,
                        interface: iface.name
                    };
                }
            }
        }
        // 如果没有下一跳IP，对于默认路由（0.0.0.0/0），使用第一个接口
        // 对于其他路由，检查是否是直连路由
        else if (bestMatch.network === '0.0.0.0' && bestMatch.subnetMask === '0.0.0.0') {
            // 默认路由，使用第一个可用接口
            if (router.properties.interfaces.length > 0) {
                return {
                    network: bestMatch.network,
                    subnetMask: bestMatch.subnetMask,
                    nextHop: bestMatch.nextHop,
                    interface: router.properties.interfaces[0].name
                };
            }
        }
        // 对于其他没有下一跳IP的路由，检查是否是直连路由
        else {
            for (const iface of router.properties.interfaces) {
                if (isSameSubnet(targetIp, iface.subnetMask, iface.ip, iface.subnetMask)) {
                    return {
                        network: bestMatch.network,
                        subnetMask: bestMatch.subnetMask,
                        nextHop: bestMatch.nextHop,
                        interface: iface.name
                    };
                }
            }
        }
    }
    
    // 如果没有找到匹配的路由，但路由表中有默认路由（0.0.0.0/0），使用默认路由
    if (!bestMatch) {
        for (const route of router.properties.routingTable) {
            if (route.network === '0.0.0.0' && route.subnetMask === '0.0.0.0') {
                // 默认路由必须有有效的下一跳
                if (!route.nextHop || route.nextHop === '0.0.0.0' || route.nextHop === 'unreachable') {
                    continue;
                }
                
                // 如果有下一跳IP，根据下一跳IP确定出接口
                // 查找连接到下一跳IP的接口
                for (const iface of router.properties.interfaces) {
                    if (isSameSubnet(route.nextHop, iface.subnetMask, iface.ip, iface.subnetMask)) {
                        return {
                            network: route.network,
                            subnetMask: route.subnetMask,
                            nextHop: route.nextHop,
                            interface: iface.name
                        };
                    }
                }
            }
        }
    }
    
    // 注意：我们已经在前面的代码中返回了带有interface信息的路由对象，所以这里不需要再返回bestMatch
    return null;
}

// 获取子网掩码的长度
function getMaskLength(mask) {
    if (!mask) return 0;
    const octets = mask.split('.').map(Number);
    const binary = octets.map(octet => octet.toString(2).padStart(8, '0')).join('');
    const zeroIndex = binary.indexOf('0');
    return zeroIndex !== -1 ? zeroIndex : 32;
}

// 检查路由配置是否正确
function checkRouting(sourceDevice, targetIp, path) {
    // 简化处理：检查路径中的每个路由器是否有合适的接口和路由配置
    for (let i = 0; i < path.length - 1; i++) {
        const currentDevice = path[i];
        const nextDevice = path[i + 1];
        
        if (currentDevice.type === 'router') {
            // 检查路由器是否有连接到下一跳的接口
            let hasValidInterface = false;
            
            // 首先检查路由表
            const matchingRoute = findMatchingRoute(currentDevice, targetIp);
            if (matchingRoute) {
                // 检查下一跳是否可达
                if (matchingRoute.nextHop) {
                    // 检查下一跳是否在直连子网中
                    for (const iface of currentDevice.properties.interfaces) {
                        for (const nextIface of nextDevice.properties.interfaces) {
                            if (isSameSubnet(matchingRoute.nextHop, iface.subnetMask, nextIface.ip, nextIface.subnetMask)) {
                                hasValidInterface = true;
                                break;
                            }
                        }
                        if (hasValidInterface) break;
                    }
                } else {
                    // 直连路由，检查下一跳是否在同一子网
                    if (nextDevice.type === 'router') {
                        // 路由器之间应该有直连的接口
                        for (const iface of currentDevice.properties.interfaces) {
                            for (const nextIface of nextDevice.properties.interfaces) {
                                // 简单检查是否在同一子网
                                if (isSameSubnet(iface.ip, iface.subnetMask, nextIface.ip, nextIface.subnetMask)) {
                                    hasValidInterface = true;
                                    break;
                                }
                            }
                            if (hasValidInterface) break;
                        }
                    } else {
                        // 路由器到终端设备或交换机
                        if (nextDevice.type === 'switch') {
                            // 路由器到交换机，交换机是透明的，不需要检查是否直接连接到目标设备
                            // 只要路由器有到目标的路由，就认为接口有效
                            hasValidInterface = true;
                        } else {
                            // 路由器到终端设备，直接连接
                            // 检查目标设备是否是目标IP
                            const targetDeviceIp = getDeviceIp(nextDevice);
                            if (targetDeviceIp === targetIp) {
                                // 检查当前路由器是否有接口与目标设备在同一子网
                                const targetDeviceMask = getDeviceSubnetMask(nextDevice);
                                if (targetDeviceMask) {
                                    for (const iface of currentDevice.properties.interfaces) {
                                        if (isSameSubnet(iface.ip, iface.subnetMask, targetDeviceIp, targetDeviceMask)) {
                                            hasValidInterface = true;
                                            break;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            } else {
                // 如果没有找到匹配的路由，检查是否是直连子网
                for (const iface of currentDevice.properties.interfaces) {
                    for (const nextIface of nextDevice.properties.interfaces) {
                        if (isSameSubnet(iface.ip, iface.subnetMask, nextIface.ip, nextIface.subnetMask)) {
                            hasValidInterface = true;
                            break;
                        }
                    }
                    if (hasValidInterface) break;
                }
            }
            
            if (!hasValidInterface) {
                return false;
            }
        }
    }
    
    return true;
}

// 安全设置元素的textContent
function safeSetTextContent(elementId, text) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = text;
    } else {
        console.warn(`元素 ${elementId} 不存在，无法设置textContent`);
    }
}

// 记录日志
function log(message) {
    const logContent = document.getElementById('log-content');
    if (!logContent) {
        console.log(`[${new Date().toLocaleTimeString()}] ${message}`);
        return;
    }
    const timestamp = new Date().toLocaleTimeString();
    const logLine = document.createElement('p');
    logLine.textContent = `[${timestamp}] ${message}`;
    logContent.appendChild(logLine);
    logContent.scrollTop = logContent.scrollHeight;
}

// 取消属性编辑
function cancelProperties() {
    // 重新选择当前设备或连接，刷新属性面板
    if (selectedDevice) {
        selectDevice(selectedDevice);
    } else if (selectedConnection) {
        selectConnection(selectedConnection);
    }
}

// 模态框拖动功能
let isModalDragging = false;
let dragStartX = 0;
let dragStartY = 0;
let modalContent;

// 初始化拖动功能
function initModalDrag() {
    modalContent = document.querySelector('.modal-content');
    if (modalContent) {
        // 将模态框头部（.modal-header）作为拖动区域
        const modalHeader = modalContent.querySelector('#modal-body .modal-header');
        if (modalHeader) {
            modalHeader.style.cursor = 'move';
            
            modalHeader.addEventListener('mousedown', startDrag);
            document.addEventListener('mousemove', drag);
            document.addEventListener('mouseup', stopDrag);
        }
        // 如果没有modal-header，不添加拖动功能
    }
}

// 显示邮件测试工具
function showEmailTool() {
    const modal = document.getElementById('modal');
    const modalBody = document.getElementById('modal-body');
    
    // 获取所有可用的计算机、笔记本和Email服务器
    const computers = devices.filter(device => device.type === 'computer' || device.type === 'laptop');
    const emailServers = devices.filter(device => device.type === 'email-server');
    
    modalBody.innerHTML = `
        <div class="modal-header">
            <h3>邮件收发测试工具</h3>
        </div>
        <div class="property-form">
            <!-- 选项卡 -->
            <div style="display: flex; margin-bottom: 10px; border-bottom: 1px solid #ddd;">
                <button type="button" id="send-tab" onclick="switchEmailTab('send')" 
                    style="padding: 10px 20px; background-color: #3498db; color: white; border: none; cursor: pointer;">
                    发送邮件
                </button>
                <button type="button" id="receive-tab" onclick="switchEmailTab('receive')" 
                    style="padding: 10px 20px; background-color: #f8f9fa; color: #333; border: none; cursor: pointer;">
                    接收邮件
                </button>
            </div>
            
            <!-- 发送邮件表单 -->
            <div id="send-form">
                <div class="form-group">
                    <label for="email-source">源设备</label>
                    <select id="email-source">
                        ${computers.map(device => `<option value="${device.id}">${device.name}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label for="email-sender">发件人</label>
                    <input type="email" id="email-sender" placeholder="sender@example.com">
                </div>
                <div class="form-group">
                    <label for="email-recipient">收件人</label>
                    <input type="email" id="email-recipient" placeholder="recipient@example.com">
                </div>
                <div class="form-group">
                    <label for="email-subject">主题</label>
                    <input type="text" id="email-subject" placeholder="邮件主题" value="邮件发送测试">
                </div>
                <div class="form-group">
                    <label for="email-content">内容</label>
                    <textarea id="email-content" placeholder="邮件内容" rows="5"></textarea>
                </div>
                <div class="form-actions">
                    <button class="btn-save" onclick="executeEmailTool()">发送邮件</button>
                    <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">取消</button>
                </div>
            </div>
            
            <!-- 接收邮件表单 -->
            <div id="receive-form" style="display: none;">
                <div class="form-group">
                    <label for="email-email-server">邮件服务器</label>
                    <select id="email-email-server">
                        ${emailServers.map(server => `<option value="${server.id}">${server.name} (${server.properties.ip})</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label for="email-username">用户名</label>
                    <input type="text" id="email-username" placeholder="输入用户名" style="width: 200px;">
                </div>
                <div class="form-group">
                    <label for="email-password">密码</label>
                    <div class="password-input-wrapper" style="width: 200px;">
                        <input type="password" id="email-password" placeholder="输入密码" style="width: 100%; padding-right: 30px;">
                        <span class="password-toggle-icon" onclick="togglePasswordVisibility('email-password')" title="显示/隐藏密码">◎</span>
                    </div>
                </div>
                <div class="form-actions">
                    <button class="btn-save" onclick="executeReceiveEmailTool()">查看邮件</button>
                    <button class="btn-cancel" onclick="document.getElementById('modal').style.display = 'none'">取消</button>
                </div>
            </div>
            
            <!-- 邮件结果显示区域 -->
            <div class="email-results" style="margin-top: 20px; padding: 15px; background-color: #f8f9fa; border-radius: 5px;">
                <div id="email-output" style="color: #2c3e50;"></div>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
    initModalDrag();
}

// 执行邮件发送
function executeEmailTool() {
    const sourceId = document.getElementById('email-source').value;
    const sender = document.getElementById('email-sender').value;
    const recipient = document.getElementById('email-recipient').value;
    const subject = document.getElementById('email-subject').value;
    const content = document.getElementById('email-content').value;
    
    const source = devices.find(device => device.id === sourceId);
    if (source && sender && recipient && subject && content) {
        const emailOutput = document.getElementById('email-output');
        emailOutput.innerHTML = '';
        
        // 记录日志
        log(`开始发送邮件: ${source.name} → ${recipient}`);
        
        // 模拟SMTP发送过程
        emailOutput.innerHTML += `<div style="color: #3498db; font-weight: bold;">正在准备发送邮件...</div>`;
        
        setTimeout(() => {
            // 解析发件人域名
            const senderDomain = sender.split('@')[1];
            const recipientDomain = recipient.split('@')[1];
            
            emailOutput.innerHTML += `<div style="color: #3498db; font-weight: bold;">正在解析收件人域名...</div>`;
            
            setTimeout(() => {
                // 查找MX记录
                let mxServer = null;
                let dnsServer = null;
                
                // 首先尝试使用源设备配置的DNS服务器
                const sourceDns = source.properties.dns;
                let candidateDnsServers = devices.filter(device => device.type === 'dns-server');
                
                if (sourceDns && candidateDnsServers.length > 0) {
                    // 查找配置的DNS服务器设备
                    dnsServer = candidateDnsServers.find(device => 
                        getDeviceIp(device) === sourceDns
                    );
                }
                
                // 如果没有配置DNS服务器或配置的DNS服务器不可用，使用所有可用的DNS服务器
                if (!dnsServer && candidateDnsServers.length > 0) {
                    dnsServer = candidateDnsServers[0];
                }
                
                // 检查源设备是否能到达DNS服务器
                if (dnsServer) {
                    const dnsServerIp = getDeviceIp(dnsServer);
                    if (dnsServerIp && !isReachable(source, dnsServerIp)) {
                        emailOutput.innerHTML += `<div style="color: #e74c3c;">✗ 发送失败: 无法连接到DNS服务器 ${dnsServerIp}，请检查网络连接</div>`;
                        log(`邮件发送失败: ${source.name} 无法连接到DNS服务器 ${dnsServerIp}`);
                        return;
                    }
                }
                
                if (dnsServer && dnsServer.properties.domains) {
                    // 查找收件人域名的MX记录
                    mxServer = dnsServer.properties.domains.find(domain => 
                        (domain.type === 'MX' && recipientDomain.endsWith(domain.name)) || 
                        (domain.type === 'A' && domain.name === recipientDomain)
                    );
                }
                
                if (mxServer) {
                    const emailServer = devices.find(device => 
                        device.type === 'email-server' && device.properties.ip === mxServer.ip
                    );
                    
                    if (emailServer) {
                        // 检查源设备是否能到达邮件服务器
                        const emailServerIp = getDeviceIp(emailServer);
                        if (!emailServerIp) {
                            emailOutput.innerHTML += `<div style="color: #e74c3c;">✗ 发送失败: 邮件服务器没有有效的IP地址</div>`;
                            log(`邮件发送失败: 邮件服务器没有有效的IP地址`);
                            return;
                        }
                        
                        if (!isReachable(source, emailServerIp)) {
                            emailOutput.innerHTML += `<div style="color: #e74c3c;">✗ 发送失败: 无法连接到SMTP服务器 ${emailServerIp}，请检查网络连接</div>`;
                            log(`邮件发送失败: ${source.name} 无法连接到SMTP服务器 ${emailServerIp}`);
                            return;
                        }
                        
                        emailOutput.innerHTML += `<div style="color: #27ae60;">✓ 找到MX记录: ${mxServer.ip}</div>`;
                        emailOutput.innerHTML += `<div style="color: #3498db; font-weight: bold;">正在连接SMTP服务器...</div>`;
                        
                        setTimeout(() => {
                            emailOutput.innerHTML += `<div style="color: #27ae60;">✓ SMTP连接成功</div>`;
                            emailOutput.innerHTML += `<div style="color: #3498db; font-weight: bold;">正在发送邮件...</div>`;
                            
                            setTimeout(() => {
                                // 查找收件人用户
                                const recipientUser = emailServer.properties.users.find(user => 
                                    user.username === recipient.split('@')[0]
                                );
                                
                                if (recipientUser) {
                                    // 创建新邮件
                                    const newEmail = {
                                        id: Date.now(),
                                        from: sender,
                                        to: recipient,
                                        subject: subject,
                                        content: content,
                                        timestamp: new Date().toLocaleString(),
                                        read: false
                                    };
                                    
                                    // 添加到用户邮箱
                                    recipientUser.emails.push(newEmail);
                                    
                                    emailOutput.innerHTML += `<div style="color: #27ae60;">✓ 邮件发送成功</div>`;
                                    emailOutput.innerHTML += `<div style="margin-top: 10px; padding: 10px; background-color: white; border: 1px solid #ddd;">`;
                                    emailOutput.innerHTML += `<p><strong>发件人:</strong> ${sender}</p>`;
                                    emailOutput.innerHTML += `<p><strong>收件人:</strong> ${recipient}</p>`;
                                    emailOutput.innerHTML += `<p><strong>主题:</strong> ${subject}</p>`;
                                    emailOutput.innerHTML += `<p><strong>内容:</strong> ${content}</p>`;
                                    emailOutput.innerHTML += `</div>`;
                                    
                                    // 记录成功日志
                                    log(`邮件发送成功: ${source.name} → ${recipient}`);
                                } else {
                                    emailOutput.innerHTML += `<div style="color: #e74c3c;">✗ 发送失败: 收件人邮箱不存在</div>`;
                                    log(`邮件发送失败: 收件人邮箱不存在`);
                                }
                            }, 1500);
                        }, 1000);
                    } else {
                        emailOutput.innerHTML += `<div style="color: #e74c3c;">✗ 发送失败: 未找到对应的邮件服务器</div>`;
                        log(`邮件发送失败: 未找到对应的邮件服务器`);
                    }
                } else {
                    emailOutput.innerHTML += `<div style="color: #e74c3c;">✗ 发送失败: 未找到MX记录</div>`;
                    log(`邮件发送失败: 未找到MX记录`);
                }
            }, 1000);
        }, 500);
    } else {
        alert('请填写所有必填字段');
    }
}

// 切换邮件测试工具选项卡
// 切换邮件测试工具选项卡
function switchEmailTab(tab) {
    const sendForm = document.getElementById('send-form');
    const receiveForm = document.getElementById('receive-form');
    const sendTab = document.getElementById('send-tab');
    const receiveTab = document.getElementById('receive-tab');
    
    if (tab === 'send') {
        sendForm.style.display = 'block';
        receiveForm.style.display = 'none';
        sendTab.style.backgroundColor = '#3498db';
        sendTab.style.color = 'white';
        receiveTab.style.backgroundColor = '#f8f9fa';
        receiveTab.style.color = '#333';
    } else {
        sendForm.style.display = 'none';
        receiveForm.style.display = 'block';
        sendTab.style.backgroundColor = '#f8f9fa';
        sendTab.style.color = '#333';
        receiveTab.style.backgroundColor = '#3498db';
        receiveTab.style.color = 'white';
    }
}

// 执行接收邮件
function executeReceiveEmailTool() {
    const serverId = document.getElementById('email-email-server').value;
    const username = document.getElementById('email-username').value;
    const password = document.getElementById('email-password').value;
    
    const emailServer = devices.find(device => device.id === serverId);
    if (emailServer) {
        const emailOutput = document.getElementById('email-output');
        emailOutput.innerHTML = '';
        
        // 记录日志
        log(`开始接收邮件: ${emailServer.name} - 用户 ${username}`);
        
        // 模拟POP3/IMAP连接
        emailOutput.innerHTML += `<div style="color: #3498db; font-weight: bold;">正在连接邮件服务器...</div>`;
        
        setTimeout(() => {
            // 验证用户
            const user = emailServer.properties.users.find(user => 
                user.username === username && user.password === password
            );
            
            if (user) {
                emailOutput.innerHTML += `<div style="color: #27ae60;">✓ 用户验证成功</div>`;
                emailOutput.innerHTML += `<div style="color: #3498db; font-weight: bold;">正在获取邮件列表...</div>`;
                
                setTimeout(() => {
                    if (user.emails.length > 0) {
                        emailOutput.innerHTML += `<div style="color: #27ae60;">✓ 成功获取 ${user.emails.length} 封邮件</div>`;
                        
                        // 显示邮件列表
                        emailOutput.innerHTML += `<div style="margin-top: 10px;">`;
                        user.emails.forEach((email, index) => {
                            emailOutput.innerHTML += `
                                <div style="margin: 10px 0; padding: 10px; background: white; border: 1px solid #ddd; border-radius: 4px;">
                                    <div style="display: flex; justify-content: space-between; align-items: center;">
                                        <h5 style="margin: 0; color: ${email.read ? '#666' : '#333'};">${index + 1}. ${email.subject}</h5>
                                        <span style="font-size: 12px; color: #999;">${email.timestamp}</span>
                                    </div>
                                    <p style="margin: 5px 0; font-size: 14px;"><strong>发件人:</strong> ${email.from}</p>
                                    <p style="margin: 5px 0; font-size: 14px;"><strong>收件人:</strong> ${email.to}</p>
                                    <div style="margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px;">
                                        <p>${email.content}</p>
                                    </div>
                                </div>
                            `;
                            
                            // 标记邮件为已读
                            email.read = true;
                        });
                        emailOutput.innerHTML += `</div>`;
                        
                        // 记录成功日志
                        log(`邮件接收成功: ${emailServer.name} - 用户 ${username}`);
                    } else {
                        emailOutput.innerHTML += `<div style="color: #666;">✓ 没有新邮件</div>`;
                        log(`没有新邮件: ${emailServer.name} - 用户 ${username}`);
                    }
                }, 1000);
            } else {
                emailOutput.innerHTML += `<div style="color: #e74c3c;">✗ 用户名或密码错误</div>`;
                log(`邮件接收失败: ${emailServer.name} - 用户名或密码错误`);
            }
        }, 1000);
    } else {
        alert('未找到指定的邮件服务器');
    }
}

// 开始拖动
function startDrag(e) {
    isModalDragging = true;
    // 获取鼠标相对于模态框的位置
    dragStartX = e.clientX - modalContent.getBoundingClientRect().left;
    dragStartY = e.clientY - modalContent.getBoundingClientRect().top;
    
    // 添加拖动样式
    modalContent.style.cursor = 'grabbing';
}

// 拖动中
function drag(e) {
    if (!isModalDragging || !modalContent) return;
    
    // 计算新的位置
    const newX = e.clientX - dragStartX;
    const newY = e.clientY - dragStartY;
    
    // 获取背景遮罩层的尺寸
    const modalOverlay = document.getElementById('modal');
    const overlayRect = modalOverlay.getBoundingClientRect();
    
    // 限制模态框在背景遮罩层内移动
    const maxX = overlayRect.width - modalContent.offsetWidth;
    const maxY = overlayRect.height - modalContent.offsetHeight;
    
    const clampedX = Math.max(0, Math.min(newX, maxX));
    const clampedY = Math.max(0, Math.min(newY, maxY));
    
    // 更新模态框位置
    modalContent.style.left = `${clampedX}px`;
    modalContent.style.top = `${clampedY}px`;
    modalContent.style.margin = '0'; // 覆盖原来的margin居中样式
    modalContent.style.transform = 'none'; // 移除translate变换，使用left和top精确定位
}

// 停止拖动
function stopDrag(e) {
    if (isModalDragging && modalContent) {
        isModalDragging = false;
        modalContent.style.cursor = 'move';
    }
}

// 页面加载完成后初始化
window.addEventListener('DOMContentLoaded', () => {
    init();
    initModalDrag();
});

// 窗口完全加载后再次居中画布，确保容器尺寸已确定
window.addEventListener('load', () => {
    centerCanvas();
});
